package com.xforceplus.tenant.security.client.feign.support;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import feign.RequestTemplate;
import feign.codec.EncodeException;
import feign.codec.Encoder;

import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;

/**
 * Provides support for encoding spring Pageable via composition.
 *
 * @author geewit
 */
public class PageableSpringEncoder implements Encoder {

    private final Encoder delegate;

    private final static String DEFAULT_SORT_PARAMETER = "sort";

    /**
     * Page index parameter name.
     */
    private String pageParameter = "page";

    /**
     * Page size parameter name.
     */
    private String sizeParameter = "size";

    /**
     * Sort parameter name.
     */
    private String sortParameter = DEFAULT_SORT_PARAMETER;

    /**
     * Whether to expose and assume 1-based page number indexes. Defaults to "false",
     * meaning a page number of 0 in the request equals the first page.
     */
    private boolean oneIndexedParameters = false;

    /**
     * Creates a new PageableSpringEncoder with the given delegate for fallback. If no
     * delegate is provided and this encoder cant handle the request, an EncodeException
     * is thrown.
     * @param delegate The optional delegate.
     */
    public PageableSpringEncoder(Encoder delegate) {
        this.delegate = delegate;
    }

    public void setPageParameter(String pageParameter) {
        this.pageParameter = pageParameter;
    }

    public void setSizeParameter(String sizeParameter) {
        this.sizeParameter = sizeParameter;
    }

    public void setSortParameter(String sortParameter) {
        this.sortParameter = sortParameter;
    }

    public void setOneIndexedParameters(boolean oneIndexedParameters) {
        this.oneIndexedParameters = oneIndexedParameters;
    }

    @Override
    public void encode(Object object, Type bodyType, RequestTemplate template)
            throws EncodeException {

        if (supports(object)) {
            if (object instanceof Pageable) {
                Pageable pageable = (Pageable) object;

                if (pageable.isPaged()) {
                    template.query(pageParameter, (oneIndexedParameters ? pageable.getPageNumber() + 1 : pageable.getPageNumber()) + "");
                    template.query(sizeParameter, pageable.getPageSize() + "");
                }
                this.applySort(template, pageable.getSort());
            }
            else if (object instanceof Sort) {
                Sort sort = (Sort) object;
                this.applySort(template, sort);
            }
        }
        else {
            if (delegate != null) {
                delegate.encode(object, bodyType, template);
            }
            else {
                throw new EncodeException(
                        "PageableSpringEncoder does not support the given object "
                                + object.getClass()
                                + " and no delegate was provided for fallback!");
            }
        }
    }

    private void applySort(RequestTemplate template, Sort sort) {
        Collection<String> existingSorts = template.queries().get(DEFAULT_SORT_PARAMETER);
        List<String> sortQueries = existingSorts != null ? new ArrayList<>(existingSorts)
                : new ArrayList<>();
        if (!DEFAULT_SORT_PARAMETER.equals(sortParameter)) {
            existingSorts = template.queries().get(sortParameter);
            if (existingSorts != null) {
                sortQueries.addAll(existingSorts);
            }
        }
        for (Sort.Order order : sort) {
            sortQueries.add(order.getProperty() + "," + order.getDirection());
        }
        if (!sortQueries.isEmpty()) {
            template.query(sortParameter, sortQueries);
        }
    }

    protected boolean supports(Object object) {
        return object instanceof Pageable || object instanceof Sort;
    }

}
