package com.xforceplus.tenant.security.client.feign.utils;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.xforceplus.tenant.core.exception.TenantFeignException;
import com.xforceplus.tenant.security.client.feign.service.ClientService;
import com.xforceplus.tenant.security.client.feign.support.*;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.tenant.security.core.domain.deserializer.OrgTypeDeserializer;
import feign.*;
import feign.codec.Decoder;
import feign.codec.Encoder;
import feign.codec.ErrorDecoder;
import feign.form.spring.SpringFormEncoder;
import feign.optionals.OptionalDecoder;
import io.geewit.core.jackson.databind.serializer.BigDecimalSerializer;
import io.geewit.core.jackson.databind.serializer.EnumNameSerializer;
import io.geewit.core.jackson.databind.serializer.EnumValueSerializer;
import io.geewit.core.jackson.databind.serializer.JsonPageSerializer;
import io.geewit.core.utils.enums.Name;
import io.geewit.web.convert.converter.DateConverter;
import io.geewit.web.convert.converter.EnumValueConverter;
import io.geewit.web.convert.converter.EnumValueToIntegerConverter;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.ObjectFactory;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.cloud.openfeign.AnnotatedParameterProcessor;
import org.springframework.cloud.openfeign.FeignClientProperties;
import org.springframework.cloud.openfeign.annotation.*;
import org.springframework.cloud.openfeign.support.ResponseEntityDecoder;
import org.springframework.cloud.openfeign.support.SpringDecoder;
import org.springframework.cloud.openfeign.support.SpringEncoder;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.EnumerablePropertySource;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.io.ResourceLoader;
import org.springframework.data.domain.Page;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.format.support.FormattingConversionService;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * @author geewit
 */
@Slf4j
public class FeignUtils {
    public static final List<AnnotatedParameterProcessor> DEFAULT_ANNOTATED_ARGUMENT_RESOLVERS = Stream.of(
            new MatrixVariableParameterProcessor(),
            new PathVariableParameterProcessor(),
            new RequestParamParameterProcessor(),
            new RequestHeaderParameterProcessor(),
            new QueryMapParameterProcessor(),
            new RequestPartParameterProcessor()).collect(Collectors.toList());

    private static Feign.Builder defaultBuilder(feign.Client client,
                                                ClientService clientService,
                                                Properties properties) {

        FormattingConversionService formattingConversionService = new DefaultFormattingConversionService();
        return Feign.builder()
                .decoder(feignDecoder())
                .errorDecoder(errorDecoder(clientService))
                .contract(feignContract(DEFAULT_ANNOTATED_ARGUMENT_RESOLVERS, formattingConversionService, properties))
                .client(client);
    }

    public static Feign.Builder of(feign.Client client, Properties properties) {
        return defaultBuilder(client, null, properties)
                .encoder(feignEncoder());
    }

    public static Feign.Builder of(feign.Client client, ClientService clientService, Properties properties) {
        return defaultBuilder(client, clientService, properties).encoder(feignEncoder());
    }

    public static Feign.Builder of(feign.Client client, ClientService clientService, ResourceLoader resourceLoader) {
        return defaultBuilder(client, clientService, toProperties(resourceLoader)).encoder(feignEncoder());
    }

    public static Feign.Builder of(feign.Client client,
                                   Properties properties,
                                   String pageParameter,
                                   String sizeParameter,
                                   String sortParameter,
                                   Boolean oneIndexedParameters) {
        return of(client, null, properties, pageParameter, sizeParameter, sortParameter, oneIndexedParameters);
    }

    public static Feign.Builder of(feign.Client client,
                                   RestTemplate tenantRestTemplate,
                                   OauthClientProperties oauthClientProperties,
                                   Properties properties,
                                   String pageParameter,
                                   String sizeParameter,
                                   String sortParameter,
                                   Boolean oneIndexedParameters) {
        ClientService clientService = new ClientService(tenantRestTemplate, oauthClientProperties);
        return of(client, clientService, properties, pageParameter, sizeParameter, sortParameter, oneIndexedParameters);
    }

    public static Feign.Builder of(feign.Client client,
                                   ClientService clientService,
                                   ResourceLoader resourceLoader,
                                   String pageParameter,
                                   String sizeParameter,
                                   String sortParameter,
                                   Boolean oneIndexedParameters) {
        Properties properties = toProperties(resourceLoader);
        Feign.Builder builder = defaultBuilder(client, clientService, properties);
        builder = builder.encoder(feignEncoderPageable(pageParameter, sizeParameter, sortParameter, oneIndexedParameters));
        return builder;
    }

    public static Feign.Builder of(feign.Client client,
                                   ClientService clientService,
                                   Properties properties,
                                   String pageParameter,
                                   String sizeParameter,
                                   String sortParameter,
                                   Boolean oneIndexedParameters) {
        return defaultBuilder(client, clientService, properties)
                .encoder(feignEncoderPageable(pageParameter, sizeParameter, sortParameter, oneIndexedParameters));
    }

    public static Feign.Builder builderOption(Feign.Builder builder, FeignClientProperties.FeignClientConfiguration defaultConifg) {
        Integer connectionTimeout = null;
        Integer readTimeout = null;
        if (defaultConifg != null) {
            connectionTimeout = defaultConifg.getConnectTimeout();
            readTimeout = defaultConifg.getReadTimeout();
        }
        if (connectionTimeout == null || connectionTimeout <= 0) {
            connectionTimeout = 10000;
        }
        if (readTimeout == null || readTimeout <= 0) {
            readTimeout = 60000;
        }
        return builder.options(new Request.Options(readTimeout, TimeUnit.MILLISECONDS, connectionTimeout, TimeUnit.MILLISECONDS, true));
    }

    public static Feign.Builder registerRequestInterceptors(Feign.Builder builder,
                                                            Iterable<RequestInterceptor> requestInterceptors,
                                                            RequestInterceptor... extendRequestInterceptors) {
        if (requestInterceptors != null && requestInterceptors.iterator().hasNext()) {
            builder.requestInterceptors(requestInterceptors);
        }
        if (extendRequestInterceptors != null) {
            Arrays.stream(extendRequestInterceptors).forEach(builder::requestInterceptor);
        }
        return builder;
    }

    public static ObjectFactory<HttpMessageConverters> feignHttpMessageConverter() {
        MappingJackson2HttpMessageConverter mappingJackson2HttpMessageConverter = new MappingJackson2HttpMessageConverter();
        ObjectMapper objectMapper = mappingJackson2HttpMessageConverter.getObjectMapper();
        Module pageJacksonModule = pageJacksonModule();
        Module sortJacksonModule = sortJacksonModule();
        objectMapper.registerModule(pageJacksonModule);
        objectMapper.registerModule(sortJacksonModule);
        SimpleModule simpleModule = new SimpleModule();
        simpleModule.addSerializer(OrgType.class, EnumValueSerializer.instanceOfInteger);
        simpleModule.addSerializer(Name.class, EnumNameSerializer.instance);
        simpleModule.addSerializer(Long.class, ToStringSerializer.instance);
        simpleModule.addSerializer(Long.TYPE, ToStringSerializer.instance);
        simpleModule.addSerializer(BigDecimal.class, BigDecimalSerializer.instance);
        simpleModule.addSerializer(Page.class, new JsonPageSerializer(objectMapper));
        simpleModule.addDeserializer(OrgType.class, OrgTypeDeserializer.instance);
        objectMapper.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
        objectMapper.disable(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_AS_NULL);
        objectMapper.disable(DeserializationFeature.FAIL_ON_MISSING_CREATOR_PROPERTIES);
        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        objectMapper.registerModule(simpleModule);
        final HttpMessageConverters httpMessageConverters = new HttpMessageConverters(mappingJackson2HttpMessageConverter);
        return () -> httpMessageConverters;
    }

    public static Decoder feignDecoder() {
        return new OptionalDecoder(
                new ResponseEntityDecoder(new SpringDecoder(feignHttpMessageConverter())));
    }

    public static ErrorDecoder errorDecoder(ClientService clientService) {
        return (methodKey, response) -> {
            if (clientService != null && response.status() == HttpStatus.UNAUTHORIZED.value()) {
                clientService.refresh();
            }
            Request request = response.request();
            if (request != null) {
                String requestUrl = request.url();
                String requestMethod = request.httpMethod().name();
                Map<String, Collection<String>> requestHeaders = request.headers();
                log.error("request url:{}, method:{}, headers:{}", requestUrl, requestMethod, JsonUtils.toJson(requestHeaders));
                if (request.body() != null) {
                    String requestBody = new String(request.body(), StandardCharsets.UTF_8);
                    log.error("request.body:{}", requestBody);
                }
            }
            String responseBody;
            if (response.body() != null) {
                try {
                    responseBody = Util.toString(response.body().asReader(StandardCharsets.UTF_8));
                    log.debug("responseBody = {}", responseBody);
                    if (response.status() >= HttpStatus.BAD_REQUEST.value() && response.status() < HttpStatus.INTERNAL_SERVER_ERROR.value()) {
                        log.error("response status: {}, body: {}", response.status(), responseBody);
                        throw new TenantFeignException("CLIENT", responseBody, response.status());
                    }
                } catch (IOException e) {
                    log.error("获取返回值发生错误");
                }
            }
            return FeignException.errorStatus(methodKey, response);
        };
    }

    public static Encoder feignEncoder() {
        return new SpringEncoder(new SpringFormEncoder(), feignHttpMessageConverter());
    }

    public static PageableSpringEncoder feignEncoderPageable(String pageParameter,
                                                             String sizeParameter,
                                                             String sortParameter,
                                                             Boolean oneIndexedParameters) {
        PageableSpringEncoder encoder = new PageableSpringEncoder(feignEncoder());
        encoder.setPageParameter(pageParameter);
        encoder.setSizeParameter(sizeParameter);
        encoder.setSortParameter(sortParameter);
        encoder.setOneIndexedParameters(oneIndexedParameters);
        return encoder;
    }

    public static PageJacksonModule pageJacksonModule() {
        return new PageJacksonModule();
    }

    public static SortJacksonModule sortJacksonModule() {
        return new SortJacksonModule();
    }

    public static void addConverters(FormattingConversionService feignConversionService) {
        feignConversionService.addConverter(String.class, OrgType.class, new EnumValueConverter<>());
        feignConversionService.addConverter(Long.class, OrgType.class, new EnumValueConverter<>());
        feignConversionService.addConverter(long.class, OrgType.class, new EnumValueConverter<>());
        feignConversionService.addConverter(OrgType.class, Integer.class, new EnumValueToIntegerConverter<>());
        feignConversionService.addConverter(String.class, Date.class, new DateConverter());
    }

    public static Contract feignContract(List<AnnotatedParameterProcessor> parameterProcessors,
                                         FormattingConversionService feignConversionService,
                                         Properties properties) {
        return new SpringMvcContract(parameterProcessors, feignConversionService, properties);
    }

    public static Properties toProperties(ResourceLoader resourceLoader) {
        Properties properties = new Properties();
        if (resourceLoader instanceof ConfigurableApplicationContext) {
            MutablePropertySources propertySources = ((ConfigurableApplicationContext) resourceLoader).getEnvironment().getPropertySources();
            for (org.springframework.core.env.PropertySource<?> propertySource : propertySources) {
                if (propertySource instanceof EnumerablePropertySource) {
                    EnumerablePropertySource<?> enumerablePropertySource = (EnumerablePropertySource<?>) propertySource;
                    Arrays.stream(enumerablePropertySource.getPropertyNames())
                            .forEach(propertyName -> properties.setProperty(propertyName, Objects.requireNonNull(enumerablePropertySource.getProperty(propertyName)).toString()));
                }
            }
        }
        return properties;
    }
}
