package com.xforceplus.tenant.security.client.feign.service;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.xforceplus.tenant.security.client.feign.support.OauthClientProperties;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;


/**
 * @author geewit
 */
public class ClientService {
    private static final Logger logger = LoggerFactory.getLogger(ClientService.class);

    public ClientService(RestTemplate restTemplate, OauthClientProperties clientProperties) {
        this.restTemplate = restTemplate;
        this.clientProperties = clientProperties;
        this.cache = CacheBuilder.newBuilder()
                .maximumSize(1)
                .expireAfterWrite(Duration.of(clientProperties.getExpireHours(), ChronoUnit.HOURS))
                .build(new CacheLoader<String, String>() {
                    @Override
                    public String load(@NotNull String key) {
                        return ClientService.this.fetchToken();
                    }
                });
        logger.info("ClientService initialized");
    }

    private final AtomicInteger counter = new AtomicInteger(0);

    private final RestTemplate restTemplate;

    private final OauthClientProperties clientProperties;

    private final LoadingCache<String, String> cache;

    public String token() {
        if(clientProperties.getParams() == null) {
            logger.warn("clientProperties.params == null, return null");
            return null;
        }
        String clientId = clientProperties.getParams().getClientId();
        if(StringUtils.isBlank(clientId)) {
            logger.warn("clientProperties.params.clientId is blank, return null");
            return null;
        }
        if(StringUtils.isBlank(clientProperties.getParams().getSecret())) {
            logger.warn("clientProperties.params.secret is blank, return null");
            return null;
        }
        logger.debug("clientProperties.params.clientId = {}", clientId);
        return cache.getUnchecked(clientId);
    }

    public void refresh() {
        logger.info("refresh cache");
        cache.cleanUp();
    }

    private String fetchToken() {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity<OauthClientProperties.Params> request = new HttpEntity<>(clientProperties.getParams(), headers);
            logger.debug("request.url = {}", clientProperties.getRequestUrl());
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(clientProperties.getRequestUrl(), request, String.class);
            if(responseEntity.getStatusCode().is2xxSuccessful()) {
                String body = responseEntity.getBody();
                Map<String, String> bodyMap = JsonUtils.fromJson(body, new TypeReference<Map<String, String>>(){});
                String token = bodyMap.get("data");
                logger.info("fetch token: {}", token);
                if(StringUtils.isEmpty(token)) {
                    return this.exceptionFetch();
                } else {
                    counter.set(0);
                    return token;
                }
            } else {
                logger.warn("requestUrl: {} failed, (reponse.status: {})", clientProperties.getRequestUrl(), responseEntity.getStatusCodeValue());
                return this.exceptionFetch();
            }
        } catch (RestClientException e) {
            logger.warn("requestUrl: {} failed, e: {}", clientProperties.getRequestUrl(), e.getMessage());
            return this.exceptionFetch();
        }
    }

    private String exceptionFetch() {
        this.refresh();
        try {
            logger.info("sleep 1s");
            Thread.sleep(clientProperties.getSleepMillis());
            logger.info("fetchToken again");
            int count = counter.getAndAdd(1);
            if(count > clientProperties.getMaxFailTimes()) {
                counter.set(0);
                String message = "发生无法解决的错误";
                logger.warn(message);
                throw new RuntimeException(message);
            }
            return this.fetchToken();
        } catch (InterruptedException e) {
            logger.warn("sleep exception: {}", e.getMessage());
            return this.fetchToken();
        }
    }
}
