package com.xforceplus.tenant.security.client.feign.utils;

import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.xforceplus.tenant.core.exception.TenantFeignException;
import com.xforceplus.tenant.security.client.feign.support.PageJacksonModule;
import com.xforceplus.tenant.security.client.feign.support.PageableSpringEncoder;
import com.xforceplus.tenant.security.client.feign.support.SortJacksonModule;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.tenant.security.core.domain.deserializer.OrgTypeDeserializer;
import feign.*;
import feign.codec.Decoder;
import feign.codec.Encoder;
import feign.codec.ErrorDecoder;
import feign.form.spring.SpringFormEncoder;
import feign.optionals.OptionalDecoder;
import io.geewit.core.jackson.databind.serializer.BigDecimalSerializer;
import io.geewit.core.jackson.databind.serializer.EnumNameSerializer;
import io.geewit.core.jackson.databind.serializer.EnumValueSerializer;
import io.geewit.core.jackson.databind.serializer.JsonPageSerializer;
import io.geewit.core.utils.enums.Name;
import io.geewit.web.convert.converter.DateConverter;
import io.geewit.web.convert.converter.EnumValueConverter;
import io.geewit.web.convert.converter.EnumValueToIntegerConverter;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ObjectFactory;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.cloud.openfeign.AnnotatedParameterProcessor;
import org.springframework.cloud.openfeign.FeignClientProperties;
import org.springframework.cloud.openfeign.annotation.*;
import org.springframework.cloud.openfeign.support.ResponseEntityDecoder;
import org.springframework.cloud.openfeign.support.SpringDecoder;
import org.springframework.cloud.openfeign.support.SpringEncoder;
import org.springframework.cloud.openfeign.support.SpringMvcContract;
import org.springframework.core.io.ResourceLoader;
import org.springframework.data.domain.Page;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.format.support.FormattingConversionService;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.TimeUnit;


/**
 * @author geewit
 */
public class FeignUtils {
    private static final Logger logger = LoggerFactory.getLogger(FeignUtils.class);
    private static final String PASSWORD="password";
    private static final String SEPARATOR ="\\?";

    private static Feign.Builder defaultBuilder(feign.Client client, ResourceLoader resourceLoader) {
        List<AnnotatedParameterProcessor> annotatedArgumentResolvers = new ArrayList<>();
        annotatedArgumentResolvers.add(new MatrixVariableParameterProcessor());
        annotatedArgumentResolvers.add(new PathVariableParameterProcessor());
        annotatedArgumentResolvers.add(new RequestParamParameterProcessor());
        annotatedArgumentResolvers.add(new RequestHeaderParameterProcessor());
        annotatedArgumentResolvers.add(new QueryMapParameterProcessor());
        annotatedArgumentResolvers.add(new RequestPartParameterProcessor());
        FormattingConversionService formattingConversionService = new DefaultFormattingConversionService();
        return Feign.builder()
                .decoder(feignDecoder())
                .errorDecoder(errorDecoder())
                .contract(feignContract(resourceLoader, annotatedArgumentResolvers, formattingConversionService))
                .client(client);
    }

    public static Feign.Builder of(feign.Client client, ResourceLoader resourceLoader) {
        Feign.Builder builder = defaultBuilder(client, resourceLoader);
        builder = builder.encoder(feignEncoder());
        return builder;
    }

    public static Feign.Builder of(feign.Client client,
                                   ResourceLoader resourceLoader,
                                   String pageParameter,
                                   String sizeParameter,
                                   String sortParameter,
                                   Boolean oneIndexedParameters) {
        Feign.Builder builder = defaultBuilder(client, resourceLoader);
        builder = builder.encoder(feignEncoderPageable(pageParameter, sizeParameter, sortParameter, oneIndexedParameters));
        return builder;
    }

    public static Feign.Builder builderOption(Feign.Builder builder, FeignClientProperties.FeignClientConfiguration defaultConifg) {
        Integer connectionTimeout = null;
        Integer readTimeout = null;
        if(defaultConifg != null) {
            connectionTimeout = defaultConifg.getConnectTimeout();
            readTimeout = defaultConifg.getReadTimeout();
        }
        if(connectionTimeout == null || connectionTimeout <= 0) {
            connectionTimeout = 10000;
        }
        if(readTimeout == null || readTimeout <= 0) {
            readTimeout = 60000;
        }
        return builder.options(new Request.Options(readTimeout, TimeUnit.MILLISECONDS, connectionTimeout, TimeUnit.MILLISECONDS, true));
    }

    public static Feign.Builder registerRequestInterceptors(Feign.Builder builder,
                                   Iterable<RequestInterceptor> requestInterceptors,
                                   RequestInterceptor... extendRequestInterceptors) {
        if (requestInterceptors != null && requestInterceptors.iterator().hasNext()) {
            builder = builder.requestInterceptors(requestInterceptors);
        }
        if (extendRequestInterceptors != null && extendRequestInterceptors.length > 0) {
            for (RequestInterceptor extendRequestInterceptor : extendRequestInterceptors) {
                builder = builder.requestInterceptor(extendRequestInterceptor);
            }
        }
        return builder;
    }

    public static ObjectFactory<HttpMessageConverters> feignHttpMessageConverter() {
        MappingJackson2HttpMessageConverter mappingJackson2HttpMessageConverter = new MappingJackson2HttpMessageConverter();
        ObjectMapper objectMapper = mappingJackson2HttpMessageConverter.getObjectMapper();
        Module pageJacksonModule = pageJacksonModule();
        Module sortJacksonModule = sortJacksonModule();
        objectMapper.registerModule(pageJacksonModule);
        objectMapper.registerModule(sortJacksonModule);
        SimpleModule simpleModule = new SimpleModule();
        simpleModule.addSerializer(OrgType.class, EnumValueSerializer.instance);
        simpleModule.addSerializer(Name.class, EnumNameSerializer.instance);
        simpleModule.addSerializer(Long.class, ToStringSerializer.instance);
        simpleModule.addSerializer(Long.TYPE, ToStringSerializer.instance);
        simpleModule.addSerializer(BigDecimal.class, BigDecimalSerializer.instance);
        simpleModule.addSerializer(Page.class, new JsonPageSerializer(objectMapper));
        simpleModule.addDeserializer(OrgType.class, OrgTypeDeserializer.instance);

        objectMapper.registerModule(simpleModule);
        final HttpMessageConverters httpMessageConverters = new HttpMessageConverters(mappingJackson2HttpMessageConverter);
        return () -> httpMessageConverters;
    }

    public static Decoder feignDecoder() {
        return new OptionalDecoder(
                new ResponseEntityDecoder(new SpringDecoder(feignHttpMessageConverter())));
    }

    public static ErrorDecoder errorDecoder() {
        return (methodKey, response) -> {
            Request request = response.request();
            if (request != null) {
                String requestUrl = request.url();
                String requestMethod = request.httpMethod().name();
                Map<String, Collection<String>> requestHeaders = request.headers();
                logger.error("request url:{}, method:{}, headers:{}", requestUrl, requestMethod, JsonUtils.toJson(requestHeaders));
                if (request.body() != null) {
                    String requestBody = new String(request.body(), StandardCharsets.UTF_8);
                    logger.error("request.body:{}", requestBody);
                }
            }
            String responseBody;
            if (response.body() != null) {
                try {
                    responseBody = Util.toString(response.body().asReader(StandardCharsets.UTF_8));
                    logger.debug("responseBody = {}", responseBody);
                    if (response.status() >= HttpStatus.BAD_REQUEST.value() && response.status() < HttpStatus.INTERNAL_SERVER_ERROR.value()) {
                        logger.error("response status: {}, body: {}", response.status(), responseBody);
                        throw new TenantFeignException("CLIENT", responseBody, response.status());
                    }
                } catch (IOException e) {
                    logger.error("获取返回值发生错误");
                }
            }
            return FeignException.errorStatus(methodKey, response);
        };
    }

    public static Encoder feignEncoder() {
        return new SpringEncoder(new SpringFormEncoder(), feignHttpMessageConverter());
    }

    public static PageableSpringEncoder feignEncoderPageable(String pageParameter,
                                                             String sizeParameter,
                                                             String sortParameter,
                                                             Boolean oneIndexedParameters) {
        PageableSpringEncoder encoder = new PageableSpringEncoder(feignEncoder());
        encoder.setPageParameter(pageParameter);
        encoder.setSizeParameter(sizeParameter);
        encoder.setSortParameter(sortParameter);
        encoder.setOneIndexedParameters(oneIndexedParameters);
        return encoder;
    }

    public static PageJacksonModule pageJacksonModule() {
        return new PageJacksonModule();
    }

    public static SortJacksonModule sortJacksonModule() {
        return new SortJacksonModule();
    }

    public static void addConverters(FormattingConversionService feignConversionService) {
        feignConversionService.addConverter(String.class, OrgType.class, new EnumValueConverter<>());
        feignConversionService.addConverter(Integer.class, OrgType.class, new EnumValueConverter<>());
        feignConversionService.addConverter(int.class, OrgType.class, new EnumValueConverter<>());
        feignConversionService.addConverter(OrgType.class, Integer.class, new EnumValueToIntegerConverter<>());
        feignConversionService.addConverter(String.class, Date.class, new DateConverter());
    }

    public static Contract feignContract(ResourceLoader resourceLoader, List<AnnotatedParameterProcessor> parameterProcessors, FormattingConversionService feignConversionService) {
        SpringMvcContract contract = new SpringMvcContract(parameterProcessors, feignConversionService);
        contract.setResourceLoader(resourceLoader);
        return contract;
    }

    private static String handleUrl(String url){
        if (StringUtils.isBlank(url)){
            return null;
        }
        String[] strings = url.split(SEPARATOR);
        if (strings.length!=2){
            return url;
        }else {
            String queryString = strings[1];
            queryString = filterSensitiveLog(queryString);
            return strings[0]+ "?" +queryString;
        }
    }

    /**
     * 将queryString中的敏感信息屏蔽掉
     *
     * @param source queryString
     * @return 去掉敏感信息的字符串
     */
    private static String filterSensitiveLog(String source) {
        if (StringUtils.isBlank(source)) {
            return null;
        } else {
            if (!source.contains(PASSWORD)) {
                return source;
            } else {
                StringBuilder sb = new StringBuilder();
                String[] elements = source.split("&");
                Arrays.stream(elements).forEach(item -> {
                    String[] kv = item.split("=");
                    if (!kv[0].equals(PASSWORD)) {
                        sb.append(item).append("&");
                    } else {
                        sb.append("pwdlength")
                                .append("=");
                        //仅有key没有value的情况
                        if (kv.length < 2) {
                            sb.append(0).append("&");
                        } else {
                            //value为“”的情况
                            if (StringUtils.isNotBlank(kv[1])) {
                                sb.append(kv[1].length());
                            } else {
                                sb.append(0);
                            }
                            sb.append("&");
                        }
                    }
                });
                return sb.toString();
            }
        }
    }

    public static void main(String[] args) {
        String url ="http://localhost:8080/api/users/get?uid=1&name=zombie&email=&phone=182177";
        url = handleUrl(url);
        System.out.println(url);
    }
}
