package com.xforceplus.tenant.security.logger.webflux.filter;

import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.logger.annotation.LoggerIgnore;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.reflect.MethodUtils;
import org.springframework.boot.web.reactive.filter.OrderedWebFilter;
import org.springframework.http.HttpMethod;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;

import java.lang.reflect.Method;

import static org.springframework.web.reactive.HandlerMapping.BEST_MATCHING_HANDLER_ATTRIBUTE;

/**
 * @author geewit
 */
@Slf4j
public class RequestLoggerFilter implements OrderedWebFilter {

    public static final int DEFAULT_ORDER = REQUEST_WRAPPER_FILTER_MAX_ORDER - 10000;

    private int order = DEFAULT_ORDER;

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, WebFilterChain chain) {
        Object handler = exchange.getAttributes().get(BEST_MATCHING_HANDLER_ATTRIBUTE);
        if(handler instanceof HandlerMethod) {
            Method method = ((HandlerMethod) handler).getMethod();
            boolean ignore;
            try {
                ignore = method.isAnnotationPresent(LoggerIgnore.class);
            } catch (Exception e) {
                ignore = false;
            }
            if (ignore) {
                return chain.filter(exchange);
            }
            RequestMapping requestMapping = MethodUtils.getAnnotation(method, RequestMapping.class, true, true);
            String actionName = null;
            if (requestMapping != null) {
                actionName = requestMapping.name();
            }
            ServerHttpRequest request = exchange.getRequest();
            String requestPath = request.getURI().getPath();
            HttpMethod httpMethod = request.getMethod();
            String requestMethod = "unknown";
            if(httpMethod != null) {
                requestMethod = request.getMethod().name();
            }
            String operatorName = "unknown";
            IAuthorizedUser authorizedUser = UserInfoHolder.get();
            if(authorizedUser != null) {
                operatorName = String.valueOf(authorizedUser.getId());
            }
            String message = String.format("actionName = %s, path = %s[%s], operator = %s", actionName, requestPath, requestMethod, operatorName);
            log.info(message);
        }
        return chain.filter(exchange);
    }

    @Override
    public int getOrder() {
        return this.order;
    }

    public void setOrder(int order) {
        this.order = order;
    }
}
