package com.xforceplus.tenant.security.server.interceptor;

import com.xforceplus.tenant.security.core.context.ClientInfoHolder;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.jwt.JwtConstants;
import com.xforceplus.tenant.security.core.jwt.JwtUtils;
import com.xforceplus.tenant.security.core.utils.CompressionUtils;
import com.xforceplus.tenant.security.core.utils.RequestUtils;
import com.xforceplus.tenant.security.server.adapter.RequestTokenUtils;
import com.xforceplus.tenant.security.starter.core.UserInfoHolderUtils;
import com.xforceplus.tenant.security.token.decoder.JwtDecoder;
import com.xforceplus.tenant.security.token.domain.UserType;
import io.geewit.web.utils.WebRequestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Enumeration;
import java.util.Map;
import java.util.Objects;

/**
 * 微服务中 拦截http请求，解析请求头中的用户信息
 *
 * @author geewit
 * @since 2019-05-13
 */
public class UserContextInterceptor implements HandlerInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(UserContextInterceptor.class);

    private final ApplicationContext applicationContext;

    private final boolean tokenDecode;

    private final String secret;

    private final String defaultUserInfoJson;


    public UserContextInterceptor(ApplicationContext applicationContext) {
        logger.info("tenant-security.UserContextInterceptor initializing...");
        this.applicationContext = applicationContext;
        this.tokenDecode = applicationContext.getEnvironment().getProperty("xforce.tenant.security.starter.token.decode.enable", Boolean.class, false);
        logger.info("tokenDecode: {}", tokenDecode);
        this.secret = applicationContext.getEnvironment().getProperty("xforce.tenant.security.jwt.secret", "my_sessionjw_tsecret_xdfdffdsdfdfs_fat");
        if (this.tokenDecode) {
            logger.info("secret: {}", secret);
        }

        this.defaultUserInfoJson = applicationContext.getEnvironment().getProperty("xforce.tenant.security.starter.interceptors.default_user_info");
        logger.info("defaultUserInfoJson: {}", defaultUserInfoJson);
        logger.info("tenant-security.UserContextInterceptor initialized");
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        if (handler instanceof ResourceHttpRequestHandler) {
            logger.debug("handler instanceof ResourceHttpRequestHandler");
            return true;
        }
        if (!(handler instanceof HandlerMethod)) {
            logger.debug("handler not instanceof HandlerMethod");
            return true;
        }
        boolean isStaticPage = RequestUtils.isStaticPage(request.getRequestURI());
        if (isStaticPage) {
            logger.debug("isStaticPage: true, return");
            return true;
        }
        if (logger.isDebugEnabled()) {
            Map<String, Enumeration<String>> headers = WebRequestUtils.getHeaders(request, UserType.USER.userinfoKey());
            Enumeration<String> userinfoHeaders = headers.get(UserType.USER.userinfoKey());
            while (userinfoHeaders.hasMoreElements()) {
                logger.debug("all header.userinfos: {}", userinfoHeaders.nextElement());
            }
        }

        String userinfo;
        if (this.tokenDecode) {
            String token = RequestTokenUtils.getToken(request);
            if (StringUtils.isBlank(token)) {
                String message = "访问失败，无效令牌";
                logger.warn("token isBlank, {}, return", message);
                return true;
            }
//            Map<String, String> claims = JwtUtils.decodeToken(token);
//            String userId = "userid";
//            //token为client类型
//            if (!CollectionUtils.isEmpty(claims) && Objects.equals(Objects.requireNonNull(claims).get(JwtConstants.TYPE_KEY), UserType.APPID.value())) {
//                if (claims.containsKey(userId)) {
//                    ClientInfoHolder.put(claims.get(userId));
//                    return true;
//                }
//            }
            userinfo = JwtDecoder.parseUserInfoFromToken(token, this.secret);
            logger.debug("userinfo from token: {}", userinfo);
        } else {
            userinfo = request.getHeader(UserType.USER.userinfoKey());
            logger.debug("userinfo from header: {}", userinfo);
        }

        String deCompressedUserInfo;
        HandlerMethod method = ((HandlerMethod) handler);
        if (StringUtils.isEmpty(userinfo)) {
            if (StringUtils.isEmpty(this.defaultUserInfoJson)) {
                String token = RequestTokenUtils.getToken(request);
                if (token != null) {
                    Map<String, String> claims = JwtDecoder.decode(token, secret,null);
                    String userId = "userid";
                    if (!CollectionUtils.isEmpty(claims) && Objects.equals(Objects.requireNonNull(claims).get(JwtConstants.TYPE_KEY), UserType.APPID.value())) {
                        if (claims.containsKey(userId)) {
                            ClientInfoHolder.put(claims.get(userId));
                            return true;
                        }
                    }
                }
                logger.debug("this.defaultUserInfoJson is empty, return");
                return true;
            }
            deCompressedUserInfo = defaultUserInfoJson;
            userinfo = CompressionUtils.encode(defaultUserInfoJson);
        } else {
            try {
                deCompressedUserInfo = CompressionUtils.decode(userinfo);
            } catch (Exception e) {
                logger.warn("发生异常:{}, return", e.getMessage());
                return false;
            }
        }
        logger.debug("deCompressedUserInfo: {}", deCompressedUserInfo);
        try {
            if (StringUtils.isNotEmpty(deCompressedUserInfo)) {
                return UserInfoHolderUtils.setUserInfoHolder(this.applicationContext, userinfo, deCompressedUserInfo, method);
            } else {
                logger.debug("deCompressedUserInfo == null, return");
                return true;
            }
        } catch (Exception e) {
            logger.error("解析用户上下文发生异常:{}, return", e.getMessage());
            return false;
        }
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
        UserInfoHolder.clearContext();
        ClientInfoHolder.clear();
        logger.info("tenant-security.UserInfoHolder.clear...");
    }
}
