package com.xforceplus.tenant.security.starter.webflux.filter;

import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.utils.CompressionUtils;
import com.xforceplus.tenant.security.core.utils.RequestUtils;
import com.xforceplus.tenant.security.starter.core.UserInfoHolderUtils;
import com.xforceplus.tenant.security.starter.webflux.adapter.RequestTokenUtils;
import com.xforceplus.tenant.security.token.decoder.JwtDecoder;
import com.xforceplus.tenant.security.token.domain.UserType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.reactivestreams.Publisher;
import org.springframework.boot.web.reactive.filter.OrderedWebFilter;
import org.springframework.context.ApplicationContext;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.reactive.result.method.annotation.RequestMappingHandlerMapping;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;

import java.util.List;


/**
 * @author geewit
 */
@Slf4j
public class TenantUserContextFilter implements OrderedWebFilter {

    public static final int DEFAULT_ORDER = REQUEST_WRAPPER_FILTER_MAX_ORDER - 10000;

    private int order = DEFAULT_ORDER;

    private final ApplicationContext applicationContext;

    public TenantUserContextFilter(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
        log.info("tenant-security.TenantUserContextFilter initialized");
    }

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, WebFilterChain chain) {
        ServerHttpRequest request = exchange.getRequest();
        String requestUri = request.getURI().toString();
        boolean isStaticPage = RequestUtils.isStaticPage(requestUri);
        log.debug("starter isStaticPage: " + isStaticPage);
        if(isStaticPage) {
            return this.result(exchange, chain, true);
        }
        if(log.isDebugEnabled()) {
            List<String> headers = request.getHeaders().get(UserType.USER.userinfoKey());
            if(headers != null) {
                for(String header : headers) {
                    log.debug("starter all header.userinfos: {}", header);
                }
            }
        }
        Boolean tokenDecode = this.applicationContext.getEnvironment().getProperty("xforce.tenant.security.starter.token.decode.enable", Boolean.class, false);
        log.info("tokenDecode: {}", tokenDecode);
        String userinfo = null;
        if(tokenDecode) {
            String token = RequestTokenUtils.getToken(request);
            if (StringUtils.isBlank(token)) {
                String message = "访问失败，无效令牌";
                log.warn("token isBlank, " + message);
                return this.result(exchange, chain, true);
            }
            String secret = applicationContext.getEnvironment().getProperty("xforce.tenant.security.jwt.secret", "my_sessionjw_tsecret_xdfdffdsdfdfs_fat");
            log.info("secret: {}", secret);
            userinfo = JwtDecoder.parseUserInfoFromToken(token, secret);
            log.debug("starter userinfo from token: " + userinfo);
        } else {
            List<String> userinfos = request.getHeaders().get(UserType.USER.userinfoKey());
            if(userinfos != null && !userinfos.isEmpty()) {
                userinfo = userinfos.stream().filter(StringUtils::isNotBlank).findFirst().orElse(null);
            }
            log.debug("starter userinfo from header: " + userinfo);
        }

        String deCompressedUserInfo;
        if (StringUtils.isEmpty(userinfo)) {
            String defaultUserInfoJson = this.applicationContext.getEnvironment().getProperty("xforce.tenant.security.starter.interceptors.default_user_info");
            log.info("defaultUserInfoJson = " + defaultUserInfoJson);
            if (StringUtils.isEmpty(defaultUserInfoJson)) {
                return this.result(exchange, chain, true);
            }
            deCompressedUserInfo = defaultUserInfoJson;
            userinfo = CompressionUtils.encode(defaultUserInfoJson);
        } else {
            try {
                deCompressedUserInfo = CompressionUtils.decode(userinfo);
            } catch (Exception e) {
                log.warn(e.getMessage());
                return this.result(exchange, chain, false);
            }
        }
        log.debug("starter deCompressedUserInfo:" + deCompressedUserInfo);
        try {
            if (StringUtils.isNotEmpty(deCompressedUserInfo)) {
                RequestMappingHandlerMapping handlerMapping = applicationContext.getBean(StringUtils.uncapitalize(RequestMappingHandlerMapping.class.getSimpleName()), RequestMappingHandlerMapping.class);
                HandlerMethod handlerMethod = (HandlerMethod) handlerMapping.getHandler(exchange).toProcessor().peek();
                if(handlerMethod == null) {
                    log.warn("HandlerMethod == null");
                }
                boolean result = UserInfoHolderUtils.setUserInfoHolder(this.applicationContext, userinfo, deCompressedUserInfo, handlerMethod);
                return this.result(exchange, chain, result);
            } else {
                log.debug("starter deCompressedUserInfo == null");
                return chain.filter(exchange);
            }
        } catch (Exception e) {
            log.error("starter 解析用户上下文发生异常", e);
            return this.result(exchange, chain, false);
        }
    }

    @Override
    public int getOrder() {
        return this.order;
    }

    public void setOrder(int order) {
        this.order = order;
    }

    Mono<Void> result(ServerWebExchange exchange, WebFilterChain chain, boolean result) {
        Mono<Void> mono;
        if(result) {
            mono = chain.filter(exchange);
        } else {
            mono = Mono.empty();
        }
        return mono.transformDeferred(this::clearHolder);
    }

    private Publisher<Void> clearHolder(Mono<Void> call) {
        return call.doFinally((done) -> UserInfoHolder.clearContext());
    }
}
