package com.xforceplus.tenant.security.token.decoder;

import com.auth0.jwt.exceptions.TokenExpiredException;
import com.xforceplus.tenant.security.core.jwt.JwtConstants;
import com.xforceplus.tenant.security.core.jwt.JwtUtils;
import com.xforceplus.tenant.security.token.domain.UserType;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Map;

/**
 * JWT token Decoder
 * @author geewit
 */
public class JwtDecoder {
    private static final Logger logger = LoggerFactory.getLogger(JwtDecoder.class);

    /**
     * 根据JWT token 解析 userInfo
     * @param token jwt
     * @param secret         默认私钥
     * @param tenantSecrets 租户定制私钥
     * @return userInfo
     */
    public static Map<String, String> decode(String token, String secret, Map<Long, String> tenantSecrets) {
        Map<String, String> claims = null;
        if (StringUtils.isNotBlank(token)) {
            try {
                claims = JwtUtils.decodeToken(token);
                if(claims != null && !claims.isEmpty()) {
                    if(tenantSecrets != null && !tenantSecrets.isEmpty()) {
                        String tenantIdVal = claims.get(JwtConstants.TENANT_SECRET_KEY);
                        if(!JwtUtils.isBlank(tenantIdVal)) {
                            try {
                                long tenantId = Long.parseLong(tenantIdVal);
                                if(tenantId > 0) {
                                    secret = tenantSecrets.getOrDefault(tenantId, secret);
                                }
                            } catch (NumberFormatException ignore) {
                            }
                        }
                    }
                    JwtUtils.verifyTokenBySerect(token, secret);
                }
            } catch (TokenExpiredException e) {
                logger.error("token过期异常TokenExpiredException,token=={}", token);
                String message = "token过期，请重新登录";
                logger.warn(message);
                return null;
            } catch (Exception e) {
                String message = "token解析失败";
                logger.error(message + ", secret = " + secret + ", token=" + token, e);
                return null;
            }
        }if (claims == null) {
            String message = "访问失败，没有登录";
            logger.warn("claims == null, " + message);
            return null;
        }
        return claims;
    }

    /**
     * 根据JWT token 解析 userInfo
     * @param token jwt
     * @param secret
     * @return userInfo
     */
    @Deprecated
    public static Map<String, String> decode(String token, String secret) {
        return decode(token, secret, Collections.emptyMap());
    }

    /**
     * 根据JWT claims 解析 userInfo
     * @param claims claims
     * @return userInfo
     */
    public static String parseUserInfoFromClaims(Map<String, String> claims) {
        if (claims == null) {
            String message = "访问失败，没有登录";
            logger.warn("claims == null, " + message);
            return null;
        }
        String systemType = claims.get(JwtConstants.TYPE_KEY);
        if (!UserType.USER.value().equals(systemType)) {
            String message = "token非法，请重新登录";
            logger.warn("claims == null, " + message);
            return null;
        }
        String encodedUserInfo = claims.get(JwtConstants.USERINFO_KEY);  //被压缩的用户信息
        if (StringUtils.isBlank(encodedUserInfo)) {
            String message = "访问失败，无效令牌";
            logger.warn("userinfo isBlank, " + message);
            return null;
        }
        return encodedUserInfo;
    }

    public static String parseUserInfoFromToken(String token, String secret) {
        Map<String, String> claims = decode(token, secret);
        return parseUserInfoFromClaims(claims);
    }
}
