package com.xforceplus.dao;

import com.xforceplus.domain.tenant.GradingRoleOrg;
import com.xforceplus.dto.org.CompanyTenantDto;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.OrgStruct;
import io.geewit.data.jpa.essential.repository.EntityGraphJpaRepository;
import io.geewit.data.jpa.essential.repository.EntityGraphJpaSpecificationExecutor;
import io.geewit.data.jpa.essential.repository.JpaBatchExecutor;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.data.jpa.repository.EntityGraph;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;


/**
 * @author geewit
 * @since 2019-12-03
 */
public interface OrgStructDao extends EntityGraphJpaRepository<OrgStruct, Long>, EntityGraphJpaSpecificationExecutor<OrgStruct>, JpaBatchExecutor<OrgStruct>, OrgStructCustomizedDao {
    @Override
    @Modifying(clearAutomatically = true)
    @Query("delete from OrgStruct o where o.orgId = :orgId")
    void deleteById(@Param("orgId") Long id);

    @EntityGraph(value = OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT, type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.orgCode = :orgCode")
    List<OrgStruct> findByTenantIdAndOrgCode(@Param("tenantId") long tenantId, @Param("orgCode") String orgCode);

    @EntityGraph(value = OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT, type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.orgCode in :orgCodes")
    List<OrgStruct> findByTenantIdAndOrgCodes(@Param("tenantId") long tenantId, @Param("orgCodes") Collection<String> orgCodes);

    @EntityGraph(value = OrgStruct.NAMED_ENTITY_GRAPH_SIMPLE, type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId and (o.parentId is null or o.parentId = 0)")
    List<OrgStruct> findRootsByTenantId(@Param("tenantId") long tenantId);

    @Query(value = "select o.org_struct_id from sys_org_struct o where o.tenant_id = :tenantId and o.status = 1 and (o.parent_id is null or o.parent_id = 0)", nativeQuery = true)
    List<Long> findRootIdsByTenantId(@Param("tenantId") long tenantId);

    @Query("select o.company from OrgStruct o where o.tenantId = :tenantId")
    List<Company> findCompaniesByTenantId(@Param("tenantId") long tenantId);

    @Query("select count(distinct o.companyId) from OrgStruct o where o.tenantId =:tenantId and o.orgType = :#{T(com.xforceplus.tenant.security.core.domain.OrgType).COMPANY} and o.companyId > 0")
    int countCompaniesByTenantId(@Param("tenantId") long tenantId);

    @Query(value = "select o.org_struct_id from sys_org_struct o where o.tenant_id = :tenantId and o.org_code = :orgCode", nativeQuery = true)
    Long findIdByTenantIdAndCode(@Param("tenantId") long tenantId, @Param("orgCode") String orgCode);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 1, o.enabledTime = current_timestamp where o.orgId = :orgId")
    int enable(@Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 0, o.disabledTime = current_timestamp where o.orgId = :orgId")
    int disable(@Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 1, o.enabledTime = current_timestamp where o.orgId = :orgId and o.tenantId = :tenantId")
    int enable(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 0, o.disabledTime = current_timestamp where o.orgId = :orgId and o.tenantId = :tenantId")
    int disable(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @Query(value = "select o.org_struct_id from sys_org_struct o where o.tenant_id = :tenantId and o.org_struct_id in :orgIds", nativeQuery = true)
    List<Long> findOrgIdsByTenantIdAndOrgIds(@Param("tenantId") long tenantId, @Param("orgIds") Collection<Long> orgIds);

    @Query(value = "select o.parent_ids from sys_org_struct o where o.org_struct_id =:orgId", nativeQuery = true)
    String findParentIdsByOrgId(@Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.companyId = :companyId, o.status = :status, o.auditStatus = :auditStatus, o.enabledTime = current_timestamp where o.orgId = :orgId")
    void approve(@Param("orgId") long orgId, @Param("companyId") long companyId, @Param("status") int status, @Param("auditStatus") int auditStatus);

    @EntityGraph(value = OrgStruct.NAMED_ENTITY_GRAPH_SIMPLE, type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId")
    List<OrgStruct> findByTenantId(@Param("tenantId") long tenantId);

    @Query(value = "select count(o.org_struct_id) from sys_org_struct o where o.parent_id = :orgId and o.status = 1", nativeQuery = true)
    long countChildren(@Param("orgId") long orgId);

    @Query(value = "select o.org_struct_id from sys_org_struct o where o.tenant_id = :tenantId and o.org_struct_id = :orgId", nativeQuery = true)
    Long findOrgIdByTenantIdAndOrgId(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.status = 1")
    List<OrgStruct> findOrgsByTenantId(@Param("tenantId") long tenantId);

    @Query(value = "select o.org_struct_id from sys_org_struct o where o.tenant_id = :tenantId and o.org_code = :orgCode", nativeQuery = true)
    List<Long> findOrgIdByTenantIdAndOrgCode(@Param("tenantId") long tenantId, @Param("orgCode") String orgCode);

    @Query(value = "select o.parent_ids from sys_org_struct o where o.org_struct_id in :orgIds", nativeQuery = true)
    Set<String> findParentIdsByOrgIds(@Param("orgIds") Collection<Long> orgIds);

    /**
     * 按组织Code
     *
     * @param orgCode 查询列表
     * @return Set<String>
     */
    @Query(value = "select o.parent_ids from sys_org_struct o where o.org_code = :orgCode", nativeQuery = true)
    Set<String> findParentIdsByOrgCode(@Param("orgCode") String orgCode);

    @Query(value = "select o.org_struct_id from sys_org_struct o where o.parent_ids like :parentIds and o.status = 1", nativeQuery = true)
    Set<Long> findOrgIdsByParentIdsLike(@Param("parentIds") String parentIds);

    @Query("select count(o) from OrgStruct o where o.parentId = :parentId and o.companyId = :companyId")
    long countCompaniesByParentIdAndCompanyId(@Param("parentId") long parentId, @Param("companyId") long companyId);

    List<OrgStruct> findByCompanyId(Long companyId);

    @Query("select o.company from OrgStruct o where o.tenantId = :tenantId and o.company.taxNum = :taxNum and o.company.companyName = :companyName")
    List<Company> findCompaniesByTenantIdAndCompanyNameAndTaxNum(@Param("tenantId") long tenantId, @Param("companyName") String companyName, @Param("taxNum") String taxNum);

    @Query("select o.company from OrgStruct o where o.tenantId = :tenantId and o.companyId = :companyId and o.orgType = :#{T(com.xforceplus.tenant.security.core.domain.OrgType).COMPANY}")
    Company findCompanyByTenantIdAndCompanyId(@Param("tenantId") long tenantId, @Param("companyId")long companyId);

    @Query("select o.orgId from OrgStruct o where o.tenantId = :tenantId and o.companyId = :companyId")
    List<Long> findIdByTenantIdAndCompanyId(@Param("tenantId") long tenantId, @Param("companyId")long companyId);

    @Modifying(clearAutomatically = true)
    @Query("update OrgStruct o set o.status = 0 where o.orgId in (:orgIds)")
    void disableByIds(@Param("orgIds") Collection<Long> ids);

    @Query("select o.parentIds from OrgStruct o where o.tenantId = :tenantId and o.orgCode in :orgCodes")
    List<String> findParentIdsByTenantIdAndOrgCodes(@Param("tenantId") long tenantId, @Param("orgCodes") Collection<String> orgCodes);

    @Query(value = "select o from OrgStruct o where o.parentIds like :parentIds")
    List<OrgStruct> findChildrenByParentIds(@Param("parentIds") String parentIds);

    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.orgId in :orgIds and o.status = 1")
    List<OrgStruct> findOrgByTenantIdAndOrgIds(@Param("tenantId") long tenantId, @Param("orgIds") Collection<Long> orgIds);

    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.orgCode in :orgCodes and o.status = 1")
    List<OrgStruct> findOrgByTenantIdAndOrgCodes(@Param("tenantId") long tenantId, @Param("orgCodes") Collection<String> orgCodes);

    @Query("select count(o) from OrgStruct o where o.tenantId = :tenantId and o.orgCode = :orgCode")
    long countByTenantIdAndOrgCode(@Param("tenantId") long tenantId, @Param("orgCode") String orgCode);

    @Query("select new org.apache.commons.lang3.tuple.MutablePair(o.orgId, count(child.orgId)) from OrgStruct o left join OrgStruct child on child.parentId = o.orgId where o.orgId in :orgIds and child.status = 1 group by o.orgId")
    List<Pair<Long, Long>> findChildrenCountMapsByOrgIds(@Param("orgIds") Collection<Long> orgIds);

    @Query("select new com.xforceplus.dto.org.CompanyTenantDto(o.tenantId, o.tenant.tenantName, o.tenant.tenantCode, o.companyId) from OrgStruct o where o.companyId = :companyId and o.status = 1 and o.tenant.status = 1 group by o.tenantId")
    List<CompanyTenantDto> findTenantInfoByCompanyId(@Param("companyId") long companyId);

    boolean existsByTenantIdAndOrgId(long tenantId, long orgId);

    @Query("select new com.xforceplus.domain.tenant.GradingRoleOrg(o.orgId, o.gradingRoleId, o.orgName) from OrgStruct o where o.gradingRoleId in :roleIds")
    List<GradingRoleOrg> findOrgsByGradingRoleIds(@Param("roleIds") Set<Long> roleIds);

    @Query("select o.parentIds from OrgStruct o where o.gradingRoleId in :roleIds")
    List<String> listParentIdsByGradingRoleIds(@Param("roleIds") Collection<Long> gradingRoleIds);

    @Modifying(clearAutomatically = true)
    @Query("update OrgStruct o set o.defaultOrgRoleId = null where o.tenantId = :tenantId and o.defaultOrgRoleId = :roleId")
    void unSetOrgStructDefaultOrgRole(@Param("tenantId") long tenantId, @Param("roleId") Long roleId);
}
