package com.xforceplus.dao.impl;

import com.xforceplus.api.model.CompanyModel.Request.*;
import com.xforceplus.dao.CompanyCustomizedDao;
import com.xforceplus.data.query.StringQuery;
import com.xforceplus.data.repository.AbstractDefaultJpaRepositoryImpl;
import com.xforceplus.domain.company.CompanyHisDTO;
import com.xforceplus.domain.company.CompanyPackage;
import com.xforceplus.dto.company.CompanyServicePackageDTO;
import com.xforceplus.entity.Company;
import com.xforceplus.query.CompanyQueryHelper;
import io.geewit.data.jpa.essential.utils.TupleQueryUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.repository.query.QueryUtils;
import org.springframework.data.support.PageableExecutionUtils;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.persistence.Tuple;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * 产品服务包Dao
 *
 * @author geewit
 */
@SuppressWarnings({"all"})
public class CompanyCustomizedDaoImpl extends AbstractDefaultJpaRepositoryImpl implements CompanyCustomizedDao {
    private final static Logger logger = LoggerFactory.getLogger(CompanyCustomizedDaoImpl.class);

    /**
     * 按公司IDs查询公司服务包
     */
    private static final String SQL_SERVICE_PACKAGE_BY_COMPANY_IDS = "select bcsr.company_id, bsp.service_package_id, bsp.service_package_name" +
            " from bss_company_service_rel bcsr, bss_service_package bsp" +
            " where bcsr.service_package_id = bsp.service_package_id" +
            " and bcsr.company_id in :companyIds";

    /**
     *
     */
    private static final int EFFECTIVE_VALID = 1;

    /**
     * 按公司IDs查询公司服务包
     */
    private static final String SQL_COMPANY_HIS_TAX_NUM_PAGE = "select bc.* from bss_company bc where bc.company_id in " +
            " (select company_id from bss_company_audit bca where" +
            " bca.is_effective=:effective and bca.tax_num = :taxNum)";

    /**
     * 查询公司历史记录
     */
    private static final String SQL_COMPANY_HIS_BY_TAX_NUM = "select bca.company_id, bca.company_name, bca.tax_num," +
            " bca.effective_date, bca.is_effective as effective from bss_company_audit bca where bca.tax_num = :taxNum";


    @Override
    public Page<CompanyPackage> findPackages(CompanyPackageQuery query, Pageable pageable) {

        StringBuilder selectBuilder = new StringBuilder("select new com.xforceplus.domain.company.CompanyPackage(csr.id, csr.tenantId, csr.tenant.tenantCode, csr.tenant.tenantName, csr.companyId, csr.company.companyName, csr.company.taxNum, csr.status, csr.createUserName, csr.createTime) from com.xforceplus.entity.CompanyServiceRel csr");
        StringBuilder countBuilder = new StringBuilder("select count(distinct csr.companyId) from com.xforceplus.entity.CompanyServiceRel csr join csr.tenant join csr.company");
        StringBuilder groupByBuilder = new StringBuilder(" group by csr.companyId");
        StringBuilder queryBuilder = new StringBuilder();
        if (query != null) {
            boolean hasParam = false;
            if (query.getCompanyId() != null) {
                queryBuilder.append(" where");
                queryBuilder.append(" csr.companyId = :companyId");
                hasParam = true;
            }
            if (StringUtils.isNotBlank(query.getTaxNum())) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.company.taxNum = :taxNum");
                hasParam = true;
            }
            if (StringUtils.isNotBlank(query.getCompanyName())) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.company.companyName like :companyName");
                hasParam = true;
            }
            if (query.getTenantId() != null) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.tenantId = :tenantId");
                hasParam = true;
            }
            if (StringUtils.isNotBlank(query.getTenantCode())) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.tenant.tenantCode = :tenantCode");
                hasParam = true;
            }
            if (StringUtils.isNotBlank(query.getTenantName())) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.tenant.tenantName like :tenantName");
                hasParam = true;
            }
            if (query.getServicePackageId() != null && query.getServicePackageId() > 0) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.servicePackage.servicePackageId = :servicePackageId");
                hasParam = true;
            }
            if (StringUtils.isNotBlank(query.getServicePackageName())) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.servicePackage.servicePackageName like :servicePackageName");
                hasParam = true;
            }
            if (query.getStatus() != null) {
                if (hasParam) {
                    queryBuilder.append(" and");
                } else {
                    queryBuilder.append(" where");
                }
                queryBuilder.append(" csr.status = :status");
            }
        }
        selectBuilder.append(queryBuilder).append(groupByBuilder).append(" order by csr.id desc");
        countBuilder.append(queryBuilder);
        String selectHql = selectBuilder.toString();
        logger.info("selectHQL = {}", selectHql);
        TypedQuery<CompanyPackage> selectQuery = entityManager.createQuery(selectHql, CompanyPackage.class);
        String countHql = countBuilder.toString();
        logger.info("countHQL = {}", countHql);
        TypedQuery<Long> countQuery = entityManager.createQuery(countHql, Long.class);
        if (query != null) {
            if (query.getCompanyId() != null) {
                selectQuery.setParameter("companyId", query.getCompanyId());
                countQuery.setParameter("companyId", query.getCompanyId());
            }
            if (StringUtils.isNotBlank(query.getTaxNum())) {
                selectQuery.setParameter("taxNum", query.getTaxNum());
                countQuery.setParameter("taxNum", query.getTaxNum());
            }
            if (StringUtils.isNotBlank(query.getCompanyName())) {
                selectQuery.setParameter("companyName", query.getCompanyName() + "%");
                countQuery.setParameter("companyName", query.getCompanyName() + "%");
            }
            if (query.getTenantId() != null) {
                selectQuery.setParameter("tenantId", query.getTenantId());
                countQuery.setParameter("tenantId", query.getTenantId());
            }
            if (StringUtils.isNotBlank(query.getTenantCode())) {
                selectQuery.setParameter("tenantCode", query.getTenantCode());
                countQuery.setParameter("tenantCode", query.getTenantCode());
            }
            if (StringUtils.isNotBlank(query.getTenantName())) {
                selectQuery.setParameter("tenantName", query.getTenantName() + "%");
                countQuery.setParameter("tenantName", query.getTenantName() + "%");
            }
            if (query.getServicePackageId() != null && query.getServicePackageId() > 0) {
                selectQuery.setParameter("servicePackageId", query.getServicePackageId());
                countQuery.setParameter("servicePackageId", query.getServicePackageId());
            }
            if (StringUtils.isNotBlank(query.getServicePackageName())) {
                selectQuery.setParameter("servicePackageName", "%" + query.getServicePackageName() + "%");
                countQuery.setParameter("servicePackageName", "%" + query.getServicePackageName() + "%");
            }
            if (query.getStatus() != null) {
                selectQuery.setParameter("status", query.getStatus());
                countQuery.setParameter("status", query.getStatus());
            }
        }
        long totalCount = countQuery.getSingleResult();
        logger.info("totalCount = {}", totalCount);
        int startIndex = pageable.getPageNumber() * pageable.getPageSize();
        if (startIndex > totalCount) {
            return new PageImpl<>(new ArrayList<>(), pageable, totalCount);
        }
        selectQuery.setFirstResult(startIndex);
        selectQuery.setMaxResults(pageable.getPageSize());
        List<CompanyPackage> list = selectQuery.getResultList();
        Page<CompanyPackage> page = new PageImpl<>(list, pageable, totalCount);
        return page;
    }

    /**
     * 根据公司ID查询信息
     *
     * @param companyIds 公司ID集合
     * @return List<CompanyServicePackageDTO>
     */
    @Override
    public List<CompanyServicePackageDTO> findCompanyServicePackageByCompanyIds(Set<Long> companyIds) {
        Assert.notEmpty(companyIds, "公司IDS集合不能为空");
        StringQuery stringQuery = StringQuery.builder().query(SQL_SERVICE_PACKAGE_BY_COMPANY_IDS)
                .predicate(true)
                .param("companyIds", companyIds).build();
        return this.findBySql(stringQuery, CompanyServicePackageDTO.class, Boolean.TRUE);
    }

    @Override
    public Page<Tuple> findTuples(Query query, Pageable pageable) {
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Tuple> tupleCriteriaQuery = criteriaBuilder.createTupleQuery();
        Root<Company> tupleRoot = tupleCriteriaQuery.from(Company.class);
        CompanyQueryHelper.queryTuplePredicate(query, tupleRoot, tupleCriteriaQuery, criteriaBuilder);
        Sort sort = pageable.isPaged() ? pageable.getSort() : Sort.unsorted();
        if (sort.isSorted()) {
            List<Order> orders = new ArrayList<>();
            if (!CollectionUtils.isEmpty(tupleCriteriaQuery.getOrderList())) {
                orders.addAll(tupleCriteriaQuery.getOrderList());
            }
            List<Order> toOrders = QueryUtils.toOrders(sort, tupleRoot, criteriaBuilder);
            if (!CollectionUtils.isEmpty(toOrders)) {
                orders.addAll(toOrders);
            }
            if (!CollectionUtils.isEmpty(orders)) {
                tupleCriteriaQuery.orderBy(orders);
            }
        }
        TypedQuery<Tuple> listQuery = entityManager.createQuery(tupleCriteriaQuery);
        if (pageable.isPaged()) {
            listQuery.setFirstResult((int) pageable.getOffset());
            listQuery.setMaxResults(pageable.getPageSize());
        }
        List<Tuple> results = listQuery.getResultList();
        CriteriaBuilder countCriteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Long> countCriteriaQuery = countCriteriaBuilder.createQuery(Long.class);
        Root<Company> countRoot = countCriteriaQuery.from(Company.class);
        CompanyQueryHelper.queryCountPredicate(query, countRoot, countCriteriaQuery, countCriteriaBuilder);
        if (pageable.isPaged()) {
            return PageableExecutionUtils.getPage(results, pageable,
                    () -> TupleQueryUtils.executeCountQuery(TupleQueryUtils.getCountQuery(entityManager, countRoot, countCriteriaQuery, countCriteriaBuilder)));
        } else {
            return new PageImpl<>(results);
        }
    }

    @Override
    public Optional<Company> findCompanyHistoryByTaxNum(String taxNum) {
        Assert.hasText(taxNum, "税号不为空");
        StringQuery stringQuery = StringQuery.builder().query(SQL_COMPANY_HIS_TAX_NUM_PAGE)
                .predicate(true)
                .param("taxNum", taxNum)
                .param("effective", EFFECTIVE_VALID)
                .build();
        List<Company> companyList = this.findBySql(stringQuery, Company.class, Boolean.TRUE);
        if (CollectionUtils.isEmpty(companyList)) {
            return Optional.empty();
        }
        return Optional.ofNullable(companyList.get(0));
    }


    /**
     * 按税号查询公司修改的履历信息
     *
     * @param taxNum   税号
     * @param pageable 分页
     * @return Page<CompanyHisDTO
     */
    @Override
    public Page<CompanyHisDTO> pageHistoryByTaxNum(String taxNum, Pageable pageable) {
        Assert.hasText(taxNum, "税号不为空");
        StringQuery stringQuery = StringQuery.builder().query(SQL_COMPANY_HIS_BY_TAX_NUM)
                .predicate(true)
                .param("taxNum", taxNum)
                .build();
        return super.pagingSqlBy(pageable, stringQuery.getQuery(), stringQuery.getParams(), CompanyHisDTO.class, Boolean.TRUE);
    }
}
