/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ResourceExtendDaoImpl.java   2020-10-20 11-26-43
 * Author: Evan
 */
package com.xforceplus.dao.impl;

import com.xforceplus.bo.ResourceQueryBo;
import com.xforceplus.dao.ResourceExtendDao;
import com.xforceplus.data.query.StringQuery;
import com.xforceplus.data.repository.AbstractDefaultJpaRepositoryImpl;
import com.xforceplus.dto.resource.ResourceDTO;
import com.xforceplus.dto.resource.ResourceServiceApiDTO;
import com.xforceplus.dto.resource.ServiceApiRouterDTO;
import com.xforceplus.dto.resource.ServicePackageDTO;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * Title:
 *
 * <p>Description:
 *
 * <p>Copyright: 2015~2020
 *
 * <p>Company/Department: xforceplus
 *
 * @author Evan <b>Creation Time:</b> 2020-10-20 11-26-43
 * @since V1.0
 */
@SuppressWarnings("all")
@Repository
public class ResourceExtendDaoImpl extends AbstractDefaultJpaRepositoryImpl
        implements ResourceExtendDao {
    /**
     * 查询资源信息
     */
    public static final String QUERY_RESOURCE = "SELECT sr.resource_id,sr.resource_name, sr.resource_code, sr.app_id, pr.resource_code AS parent_code,sr.parent_id  " +
            " FROM sys_resource sr LEFT JOIN sys_resource pr ON sr.parent_id = pr.resource_id where 1=1 ";
    /**
     * 资源码与服务API的关系
     */
    public static final String QUERY_RESOURCE_SERVICE_API = "SELECT ssa.service_api_id,  "
            + "  ssa.service_api_name,  "
            + "  ssa.service_api_path,  "
            + "  ssa.service_api_url,  "
            + "  ssa.route_id,  "
            + "  sr.app_id,  "
            + "  ssa.request_method,  "
            + "  sr.resource_code,  "
            + "  sr.resource_id   "
            + " FROM  "
            + "  sys_service_api ssa  "
            + "  INNER JOIN sys_resource_api_rel srar ON ssa.service_api_id = srar.service_api_id  "
            + "  INNER JOIN sys_resource sr ON srar.resource_id = sr.resource_id where 1=1 ";

    /**
     * 查询所有路由信息
     *
     * @return List<ServiceApiRouterDTO>
     */
    @Override
    public List<ServiceApiRouterDTO> findRouteAll() {
        StringQuery query = StringQuery.builder().query("select route_id,path from bss_route").build();
        return this.findBySql(query, ServiceApiRouterDTO.class, Boolean.TRUE);
    }

    /**
     * 分页查询
     *
     * @param pageable 分页对象
     * @param queryBo  分页查询条件
     * @return Page<Resource>
     */
    @Override
    public Page<ResourceDTO> pagingBy(Pageable pageable, ResourceQueryBo queryBo) {
        StringQuery query = StringQuery.builder().query(QUERY_RESOURCE);
        //判断不能等于空的情况
        if (!CollectionUtils.isEmpty(queryBo.getResourceIds())) {

            query.predicateNotEmpty(queryBo.getResourceIds())
                    .query("  and sr.resource_id in(:resourceIds)")
                    .inParam("resourceIds", queryBo.getResourceIds())
                    .build();
        } else {
            query = query.predicateHasText(queryBo.getResourceName())
                    .query(" and  sr.resource_name like :resourceName")
                    .likeParam("resourceName", queryBo.getResourceName())
                    .predicateHasText(queryBo.getResourceCode())
                    .query(" and  sr.resource_code like :resourceCode")
                    .likeParam("resourceCode", queryBo.getResourceCode())
                    .predicateNotNull(queryBo.getAppId())
                    .query(" and sr.app_id=:appId")
                    .param("appId", queryBo.getAppId())
                    .build();
        }
        // 使用下划线转换器
        return super.pagingSqlBy(
                pageable, query.getQuery(), query.getParams(), ResourceDTO.class, Boolean.TRUE);
    }

    /**
     * 服务API分页查询
     *
     * @param pageable 分页查询条件
     * @param queryBo  分页对象
     * @return Page<ServiceApiDTO>
     */
    @Override
    public Page<ResourceServiceApiDTO> pagingByServiceApi(
            Pageable pageable, ResourceQueryBo queryBo) {
        StringQuery query = StringQuery.builder().query(QUERY_RESOURCE_SERVICE_API);
        if (!CollectionUtils.isEmpty(queryBo.getResourceIds())) {
            query = query.predicateNotEmpty(queryBo.getResourceIds())
                    .query("  and sr.resource_id in(:resourceIds)")
                    .inParam("resourceIds", queryBo.getResourceIds())
                    .build();
        } else {
            query = query.predicateHasText(queryBo.getResourceName())
                    .query(" and  sr.resource_name like :resourceName")
                    .likeParam("resourceName", queryBo.getResourceName())
                    .predicateHasText(queryBo.getResourceCode())
                    .query(" and  sr.resource_code like :resourceCode")
                    .likeParam("resourceCode", queryBo.getResourceCode())
                    .predicateNotNull(queryBo.getAppId())
                    .query(" and sr.app_id=:appId")
                    .param("appId", queryBo.getAppId())
                    .build();
        }
        // 使用下划线转换器
        return super.pagingSqlBy(
                pageable, query.getQuery(), query.getParams(), ResourceServiceApiDTO.class, Boolean.TRUE);
    }

    @Override
    public List<ResourceDTO> getUserRoleResourceCode(Long userId) {
        if (userId == null || userId <= 0) {
            throw new IllegalArgumentException("userId 不合法");
        }
        // 不能用分页，需要一次查询所有的，理论上一个用户不会有太多资源码
        StringQuery query =
                StringQuery.builder()
                        .query(
                                "select distinct sr.resource_id, sr.resource_code "
                                        + "from sys_role_user_rel srur "
                                        + "inner join sys_role_resourceset_rel srrr on srur.role_id = srrr.role_id "
                                        + "inner join sys_resourceset_resource_rel srsrr on srsrr.resourceset_id = srrr.resouseset_id "
                                        + "inner join sys_resource sr on sr.resource_id = srsrr.resource_id "
                                        + "where 1=1 ")
                        .predicate(userId > 0)
                        .query("  and srur.user_id =:userId")
                        .param("userId", userId)
                        .build();
        // 使用下划线转换器
        return super.findBySql(query, ResourceDTO.class, true);
    }

    @Override
    @Transactional(readOnly = true)
    public List<ServicePackageDTO> getServicePackIdByUserIdAndAppId(Long userId, Long appId) {
        if (userId == null || userId <= 0) {
            throw new IllegalArgumentException("userId 不合法");
        }
        if (appId == null || appId <= 0) {
            throw new IllegalArgumentException("appId 不合法");
        }
        StringQuery query =
                StringQuery.builder()
                        .query(
                                "  select  distinct  bcsr.service_package_id\n"
                                        + " from   sys_org_user_rel sour \n"
                                        + " inner join sys_org_struct sos on sour.org_struct_id = sos.org_struct_id  \n"
                                        + " inner join bss_company bc on bc.company_id  = sos.company_id  \n"
                                        + " inner join bss_company_service_rel bcsr on bcsr.company_id = bc.company_id  \n"
                                        + " inner join bss_service_package  bsp on bsp.service_package_id = bcsr.service_package_id "
                                        + " where 1=1 ")
                        .predicate(userId > 0)
                        .query(" and sour.user_id =:userId")
                        .param("userId", userId)
                        .predicate(appId > 0)
                        .query(" and bsp.app_id =:appId")
                        .param("appId", appId)
                        .build();
        return super.findBySql(query, ServicePackageDTO.class, true);
    }

    @Override
    @Transactional(readOnly = true)
    public List<ResourceDTO> getUserCompanyResourceByPackageIdList(List<Long> servicePackageList) {
        if (CollectionUtils.isEmpty(servicePackageList)) {
            throw new IllegalArgumentException("servicePackageList 为空！");
        }
        StringQuery query =
                StringQuery.builder()
                        .query(
                                "  select distinct sr.resource_id, sr.resource_code \n"
                                        + " from  sys_resourceset_resource_rel srsrr\n"
                                        + " inner join  bss_service_resourceset_rel bsrr on srsrr.resourceset_id = bsrr.resourceset_id  \n"
                                        + " inner join sys_resource sr on sr.resource_id = srsrr.resource_id  \n"
                                        + " where 1=1 ")
                        // bsrr.service_package_id in ")
                        .predicate(!CollectionUtils.isEmpty(servicePackageList))
                        .query(" and bsrr.service_package_id in (:servicePackageList) ")
                        .inParam("servicePackageList", servicePackageList)
                        .build();
        return super.findBySql(query, ResourceDTO.class, true);
    }
}
