/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: NamedParams.java   2020-10-20 10-46-52
 * Author: Evan
 */
package com.xforceplus.data.query;

import com.xforceplus.utils.DateUtils;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import static com.xforceplus.api.utils.Separator.LIKE;

/**
 * <p>
 * Title: 查询参数
 * </p>
 * <p>
 * Description: 查询参数
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 10-46-52
 * @since V1.0
 */
public class NamedParams {
    /**
     * Like 参数值必须是字符串：{@value}
     */
    public static final String QUERY_VALUE_LIKE_TEXT_ERROR = "Like paramValue must has text";
    /**
     * 参数NAME不能为空:{@value}
     */
    private static final String QUERY_NAME_NULL_ERROR = "Query paramName must not be null!";
    /**
     * 参数值不能为空消息
     */
    private static final String QUERY_VALUE_NULL_ERROR = "Query paramValue must not be null!";
    /**
     * 参数Date不正确:{@value}
     */
    private static final String QUERY_VALUE_DATE_NULL_ERROR = "Query paramName must not be null!";
    /***
     * 参数
     */
    private final Map<String, Object> parameters = new ConcurrentHashMap<>();

    /**
     * 默认构建函数
     */
    private NamedParams() {
        super();
    }

    /**
     * 构建单全对象
     *
     * @return NamedParams
     */
    public static NamedParams instance() {
        return new NamedParams();
    }

    /**
     * 判断是否为空
     *
     * @return Boolean
     */
    public boolean isEmpty() {
        return parameters.isEmpty();
    }

    /**
     * 获取所有参数
     *
     * @return Map<String, Object>
     */
    public Map<String, Object> getParameters() {
        return this.parameters;
    }

    /**
     * 设置参数
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     * @return NamedParams
     */
    public NamedParams param(String paramName, Object paramValue) {
        Assert.notNull(paramName, QUERY_NAME_NULL_ERROR);
        Assert.notNull(paramValue, QUERY_VALUE_NULL_ERROR);
        parameters.put(paramName, paramValue);
        return this;
    }


    /**
     * 设置参数
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     * @return NamedParams
     */
    public NamedParams paramDateStart(String paramName, Object paramValue) {
        Assert.notNull(paramName, QUERY_NAME_NULL_ERROR);
        Assert.notNull(paramValue, QUERY_VALUE_NULL_ERROR);
        Assert.isInstanceOf(Date.class, paramValue, QUERY_VALUE_DATE_NULL_ERROR);
        paramValue = DateUtils.getDateByDayBegin((Date) paramValue);
        parameters.put(paramName, paramValue);
        return this;
    }

    /**
     * 设置参数
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     * @return NamedParams
     */
    public NamedParams paramDateEnd(String paramName, Object paramValue) {
        Assert.notNull(paramName, QUERY_NAME_NULL_ERROR);
        Assert.notNull(paramValue, QUERY_VALUE_NULL_ERROR);
        Assert.isInstanceOf(Date.class, paramValue, QUERY_VALUE_DATE_NULL_ERROR);
        paramValue = DateUtils.getDateByDayEnd((Date) paramValue);
        parameters.put(paramName, paramValue);
        return this;
    }

    /**
     * like %参数%
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     * @return NamedParams NamedParams
     */
    public NamedParams likeParam(String paramName, Object paramValue) {
        this.validate(paramName, paramValue);
        parameters.put(paramName, LIKE + this.likeValidate(paramName, paramValue) + LIKE);
        return this;
    }

    /**
     * Like Start Param
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     * @return NamedParams NamedParams
     */
    public NamedParams likeStartParam(String paramName, Object paramValue) {
        this.validate(paramName, paramValue);
        parameters.put(paramName, this.likeValidate(paramName, paramValue) + LIKE);
        return this;
    }

    /**
     * 校验值
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     */
    protected void validate(String paramName, Object paramValue) {
        Assert.notNull(paramName, QUERY_NAME_NULL_ERROR);
        Assert.notNull(paramValue, QUERY_VALUE_NULL_ERROR);
    }

    /**
     * 校验Like参数
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     * @return String
     */
    protected String likeValidate(String paramName, Object paramValue) {
        this.validate(paramName, paramValue);
        String value = String.valueOf(paramValue);
        Assert.hasText(value, QUERY_VALUE_LIKE_TEXT_ERROR);
        return value;
    }

    /**
     * Like End Param
     *
     * @param paramName  参数名
     * @param paramValue 参数值
     * @return NamedParams
     */
    public NamedParams likeEndParam(String paramName, Object paramValue) {
        this.validate(paramName, paramValue);
        parameters.put(paramName, LIKE + this.likeValidate(paramName, paramValue));
        return this;
    }

    /***
     * 集合类型的参数
     * @param paramName 参数名
     * @param paramValue  参数值
     * @return NamedParams
     */
    public NamedParams inParam(String paramName, Object paramValue) {
        this.validate(paramName, paramValue);
        if (paramValue.getClass().isArray() || paramValue instanceof Collection) {
            parameters.put(paramName, paramValue);
        } else {
            List paramList = new ArrayList(2);
            paramList.add(paramValue);
            parameters.put(paramName, paramList);
        }
        return this;
    }


}
