/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: StringQueryUtils.java   2020-10-20 11-07-57
 * Author: Evan
 */
package com.xforceplus.data.query;

import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.xforceplus.api.utils.Separator.EMPTY;
import static java.util.regex.Pattern.CASE_INSENSITIVE;
import static java.util.regex.Pattern.compile;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 11-07-57
 * @since V1.0
 */
public class StringQueryUtils {
    public static final String COUNT_QUERY_STRING = "select count(%s) from %s x";

    /**
     * Returns the query string for the given class name.
     *
     * @param template
     * @param entityName
     * @return
     */
    public static String queryString(String template, String entityName) {

        Assert.hasText(entityName, QueryUtils.ENTITY_NAME_NOT_BE_NULL);

        return String.format(template, entityName);
    }

    /***
     *
     * @param entityName
     * @return String
     */
    public static String countQueryString(String entityName) {

        String countQuery = String.format(COUNT_QUERY_STRING, "x", "%s");

        return queryString(countQuery, entityName);
    }

    public static String genCountQueryString(String queryString) {
        return "select count(*) " + removeSelect(queryString);
    }

    /**
     * <pre>
     * 去除JPQL语句前的select部分，用来生成查询总记录条数的HQL语句。
     *
     * <strong>程序范例：</strong>
     * String queryCountString = "select count(*) " + QueryUtils.removeSelect(queryString);
     *
     * </pre>
     *
     * @param queryString
     * @return
     */
    public static String removeSelect(String queryString) {
        Assert.hasText(queryString, QueryUtils.QUERY_NOT_BE_NULL);
        queryString = removeFetch(queryString);
        queryString = removeOrderBy(queryString);
        int beginPos = queryString.toLowerCase().indexOf("from");
        Assert.isTrue(beginPos != -1, " the jpql : " + queryString + " must has a keyword 'from'");
        return queryString.substring(beginPos);
    }

    /**
     * 去掉orderBy语句,，用来生成查询总记录条数的HQL语句。
     *
     * @param queryString
     * @return
     */
    public static String removeOrderBy(String queryString) {
        Assert.hasText(queryString, QueryUtils.QUERY_NOT_BE_NULL);
        queryString = removeFetch(queryString);
        int beginPos = queryString.toLowerCase().indexOf("order by ");
        if (beginPos == -1) {
            return queryString;
        }
        return queryString.substring(0, beginPos);
    }

    /**
     * <pre>
     * 去除HQL语句后的order by部分
     *
     * <strong>程序范例：</strong>
     * queryCountString = HqlUtils.removeOrders(queryCountString);
     *
     * </pre>
     *
     * @param queryString
     * @return
     */
    public static String removeOrders(String queryString) {
        Pattern pattern = compile("order\\s*by[\\w|\\W|\\s|\\S]*", CASE_INSENSITIVE);
        Matcher matcher = pattern.matcher(queryString);
        StringBuffer stringBuffer = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(stringBuffer, EMPTY);
        }
        matcher.appendTail(stringBuffer);
        return stringBuffer.toString();
    }

    /**
     * <pre>
     * 去除JPQL语句内的fetch部分
     *
     * <strong>程序范例：</strong>
     * queryString = removeFetch(queryString);
     *
     * </pre>
     *
     * @param queryString
     * @return
     */
    public static String removeFetch(String queryString) {
        Assert.hasText(queryString, QueryUtils.QUERY_NOT_BE_NULL);
        return StringUtils.delete(queryString, " fetch");
    }
}
