package com.xforceplus.query;

import com.xforceplus.api.model.OrgModel.Request.Query;
import com.xforceplus.entity.*;
import com.xforceplus.tenant.security.core.domain.OrgType;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.util.CollectionUtils;

import javax.persistence.Tuple;
import javax.persistence.criteria.*;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author geewit
 */
@SuppressWarnings("all")
public class OrgQueryHelper {
    public static Specification<OrgStruct> querySpecification(Query query) {
        Specification<OrgStruct> specification = (Specification<OrgStruct>) (root, criteriaQuery, builder) -> {
            return toPredicate(query, root, criteriaQuery, builder);
        };
        return specification;
    }

    public static Predicate queryTuplePredicate(Query query, Root<OrgStruct> root, CriteriaQuery<Tuple> criteriaQuery, CriteriaBuilder builder) {
        return toPredicate(query, root, criteriaQuery, builder);
    }

    public static Predicate queryCountPredicate(Query query, Root<OrgStruct> root, CriteriaQuery<Long> criteriaQuery, CriteriaBuilder builder) {
        return toPredicate(query, root, criteriaQuery, builder);
    }

    private static <T> Predicate toPredicate(Query query, Root<OrgStruct> root, CriteriaQuery<T> criteriaQuery, CriteriaBuilder builder) {
        List<Predicate> predicates = new ArrayList<>();
        Class<T> resultType = criteriaQuery.getResultType();
        boolean isCount = resultType.isAssignableFrom(Long.class);
        boolean joinTable = false;
        Join<OrgStruct, Company> joinCompany = null;
        Join<OrgStruct, Tenant> joinTenant = null;
        List<Selection<?>> selections = null;
        if (resultType.isAssignableFrom(Tuple.class)) {
            joinCompany = root.join("company", JoinType.LEFT);
            joinTenant = root.join("tenant", JoinType.LEFT);
            selections = Stream.of(root.alias("org"), joinCompany.alias("company"), joinTenant.alias("tenant")).collect(Collectors.toList());
        }
        //region 关联User或Account
        if ((query.getUserId() != null && query.getUserId() > 0) || (query.getAccountId() != null && query.getAccountId() > 0)) {
            ListJoin<OrgStruct, OrgUserRel> joinOrgUserRel = root.joinList("orgUserRels", JoinType.LEFT);
            joinTable = true;
            if (query.getUserId() != null && query.getUserId() > 0) {
                //是否增加 where sys_org_user_rel.user_id = {userId} 语句
                boolean filterUserId = true;
                //是否 join sys_org_user_rel 增加on userId = {userId} 需要判断
                boolean joinUserWithOnCondition = false;
                //region Tuple模式
                if (selections != null) {
                    //Tuple模式下已经主动join了 company, tenant表
                    //region 如果 被用户绑定的靠前 || 返回用户绑定的勾选状态 || 返回关联租户的公司组织
                    if ((query.getUserBoundFront() != null && query.getUserBoundFront())
                            || (query.getWithUserBoundFlag() != null && query.getWithUserBoundFlag())) {
                        //region 判断是否返回用户绑定的勾选状态
                        joinOrgUserRel.on(builder.equal(joinOrgUserRel.<Long>get("userId"), query.getUserId()));
                        joinUserWithOnCondition = true;
                        //count(sys_org_user_rel.id)
                        Expression<Long> countExpression = builder.count(joinOrgUserRel.<Long>get("id"));
                        //endregion
                        //返回用户绑定的勾选状态把 Tuple模式下主动join company, tenant表, count(sys_org_user_rel.id) as userBound 的内容拼在select语句里
                        if (query.getWithUserBoundFlag() != null && query.getWithUserBoundFlag()) {
                            selections.add(countExpression.alias("userBound"));
                        }

                        //region 被用户绑定的靠前
                        if (query.getUserBoundFront() != null && query.getUserBoundFront()) {
                            // 增加 order by count(sys_org_user_rel.id)
                            criteriaQuery.orderBy(builder.desc(countExpression));
                        }
                        //endregion

                        //join sys_org_user_rel rel on rel.user_id = {userId} 条件激活时不使用 where rel.user_id = {userId}
                        filterUserId = false;
                    }
                    //endregion
                }
                //endregion

                //region 未被用户绑定的
                if (query.getUserUnBound() != null && query.getUserUnBound()) {
                    if (!joinUserWithOnCondition) {
                        joinOrgUserRel.on(builder.equal(joinOrgUserRel.<Long>get("userId"), query.getUserId()));
                        joinUserWithOnCondition = true;
                    }

                    filterUserId = false;
                    predicates.add(builder.isNull(joinOrgUserRel.<Long>get("id")));
                }
                //endregion
                //region 增加 where sys_org_user_rel.user_id = {userId} 语句
                if (filterUserId) {
                    predicates.add(builder.equal(joinOrgUserRel.<Long>get("userId"), query.getUserId()));
                }
                //endregion
            }

            //region 查询 accountId 需要join sys_org_user_rel 和 sys_saas_accounts
            if (query.getAccountId() != null && query.getAccountId() > 0) {
                Join<OrgUserRel, User> joinUser = joinOrgUserRel.join("user", JoinType.LEFT);
                predicates.add(builder.equal(joinUser.<Long>get("accountId"), query.getAccountId()));
                if (query.getStatus() != null && query.getStatus() == 1) {
                    predicates.add(builder.equal(joinUser.get("status"), 1));
                }
            }
            //endregion
        }
        //endregion

        if (selections != null) {
            //Tuple模式下已经主动join了 company表
            //region 判断是否返回是否所属租户组织
            if (query.getWithIsHost() != null && query.getWithIsHost()) {
                selections.add(builder.<Boolean>selectCase()
                        .when(builder.equal(joinCompany.<Long>get("hostTenantId"), root.<Long>get("tenantId")), Boolean.TRUE)
                        .otherwise(Boolean.FALSE).alias("thisIsHost"));
            }
            //endregion
            criteriaQuery = criteriaQuery.multiselect(selections);
        }
        if (StringUtils.isNotBlank(query.getCompanyNo())) {
            SetJoin<OrgStruct, String> joinCompanyNo = root.joinSet("companyNos", JoinType.LEFT);
            Set<String> companyNos = Arrays.stream(StringUtils.split(query.getCompanyNo(), ","))
                    .filter(StringUtils::isNotBlank).collect(Collectors.toSet());
            if (!companyNos.isEmpty()) {
                if(companyNos.size() == 1) {
                    predicates.add(builder.equal(joinCompanyNo, companyNos.stream().findFirst().get()));
                } else {
                    predicates.add(joinCompanyNo.in(companyNos));
                }
            } else {
                predicates.add(builder.equal(joinCompanyNo, query.getCompanyNo()));
            }
            joinTable = true;
        }
        Set<Long> orgIdsQuery;
        if (query.getOrgId() != null && query.getOrgId() > 0) {
            orgIdsQuery = Stream.of(query.getOrgId()).collect(Collectors.toSet());
        } else if (query.getOrgIds() != null && !query.getOrgIds().isEmpty()) {
            orgIdsQuery = new HashSet<>(query.getOrgIds());
        } else {
            orgIdsQuery = null;
        }
        if(orgIdsQuery != null && !orgIdsQuery.isEmpty()) {
            if(orgIdsQuery.size() == 1) {
                predicates.add(builder.equal(root.<Long>get("orgId"), orgIdsQuery.stream().findFirst().get()));
            } else {
                predicates.add(root.<Long>get("orgId").in(orgIdsQuery));
            }
        }
        if (query.getTenantId() != null && query.getTenantId() > 0) {
            predicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
            if (!CollectionUtils.isEmpty(query.getFilterParentIds())) {
                Set<String> filterParentIdsSet = new HashSet<>();
                List<Predicate> predicateList = new ArrayList<>();
                List<String> filterParentIdsList = new ArrayList<>(query.getFilterParentIds());
                Collections.sort(filterParentIdsList);
                for (final String filterParentIds : filterParentIdsList) {
                    if (filterParentIdsSet.stream().noneMatch(parentIds -> StringUtils.startsWith(filterParentIds, parentIds))) {
                        filterParentIdsSet.add(filterParentIds);
                        predicateList.add(builder.like(root.<String>get("parentIds"), StringUtils.appendIfMissing(filterParentIds, "%")));
                    }
                }
                if (!predicateList.isEmpty()) {
                    if (predicateList.size() == 1) {
                        predicates.add(predicateList.stream().findFirst().get());
                    } else {
                        predicates.add(builder.or(predicateList.stream().toArray(Predicate[]::new)));
                    }
                }
            }
        }
        if (StringUtils.isNotBlank(query.getTenantCode()) || StringUtils.isNotBlank(query.getTenantName()) || StringUtils.isNotBlank(query.getTenantNameEqual())) {
            if (joinTenant == null) {
                joinTenant = root.join("tenant", JoinType.LEFT);
            }
            if (StringUtils.isNotBlank(query.getTenantCode())) {
                predicates.add(builder.equal(joinTenant.<String>get("tenantCode"), query.getTenantCode()));
            }
            if (StringUtils.isNotBlank(query.getTenantNameEqual())) {
                predicates.add(builder.equal(joinTenant.<String>get("tenantName"), query.getTenantNameEqual()));
            } else if (StringUtils.isNotBlank(query.getTenantName())) {
                String tenantName = StringUtils.appendIfMissing(query.getTenantName(), "%");
                predicates.add(builder.like(joinTenant.get("tenantName"), tenantName));
            }

        }
        if (query.getCompanyId() != null && query.getCompanyId() > 0) {
            predicates.add(builder.equal(root.<Long>get("companyId"), query.getCompanyId()));
        }
        if (StringUtils.isNotBlank(query.getOrgCode())) {
            Set<String> orgCodesQuery = Arrays.stream(StringUtils.split(query.getOrgCode(), ",")).collect(Collectors.toSet());
            if (!CollectionUtils.isEmpty(orgCodesQuery)) {
                if (orgCodesQuery.size() == 1) {
                    predicates.add(builder.equal(root.<String>get("orgCode"), orgCodesQuery.stream().findFirst().get()));
                } else {
                    predicates.add(root.<String>get("orgCode").in(orgCodesQuery));
                }
            }
        }
        //region 适配 orgType 是枚举或int
        if (query.getOrgType() != null) {
            OrgType orgType;
            try {
                int orgTypeVal = Integer.parseInt(query.getOrgType());
                orgType = OrgType.values()[orgTypeVal];
            } catch (Exception e) {
                try {
                    orgType = OrgType.valueOf(query.getOrgType());
                } catch (Exception e1) {
                    orgType = null;
                }
            }
            if (orgType != null) {
                predicates.add(builder.equal(root.<OrgType>get("orgType"), orgType));
            }
        }
        //endregion

        //region 不需要根节点
        if (query.getWithNoRootOrg() != null && query.getWithNoRootOrg().equals(Boolean.TRUE)) {
            predicates.add(builder.notEqual((root.<OrgType>get("orgType")), OrgType.GROUP));
        }
        //endregion

        if (StringUtils.isNotBlank(query.getOrgNameEqual())) {
            predicates.add(builder.equal(root.<String>get("orgName"), query.getOrgNameEqual()));
        } else if (StringUtils.isNotBlank(query.getOrgName())) {
            Set<String> orgNames = Arrays.stream(StringUtils.split(query.getOrgName(), ","))
                    .filter(StringUtils::isNotBlank).collect(Collectors.toSet());
            if (!orgNames.isEmpty()) {
                if (orgNames.size() == 1) {
                    String orgName = StringUtils.appendIfMissing(query.getOrgName(), "%");
                    predicates.add(builder.like(root.<String>get("orgName"), orgName));
                } else {
                    predicates.add(root.<String>get("orgName").in(orgNames));
                }
            }
        } else if (StringUtils.isNotBlank(query.getOrgNameLike())) {
            String[] orgNameLikes = StringUtils.split(query.getOrgNameLike(), ",");
            if(ArrayUtils.isNotEmpty(orgNameLikes)) {
                List<Predicate> predicateList = Arrays.stream(orgNameLikes)
                        .map(orgName -> StringUtils.appendIfMissing(orgName, "%"))
                        .map(orgName -> StringUtils.prependIfMissing(orgName, "%"))
                        .map(orgName -> builder.like(root.get("orgName"), orgName))
                        .collect(Collectors.toList());
                if(!predicateList.isEmpty()) {
                    if(predicateList.size() == 1) {
                        predicates.add(predicateList.stream().findFirst().get());
                    } else {
                        predicates.add(builder.or(predicateList.stream().toArray(Predicate[]::new)));
                    }
                }
            }
        }

        // region 适配 orgBizType
        if(StringUtils.isNotBlank(query.getOrgBizType())) {
            String orgBizType = StringUtils.appendIfMissing(query.getOrgBizType(), "%");
            predicates.add(builder.like(root.<String>get("orgBizType"), orgBizType));
        }
        // endregion
        if (StringUtils.isNotBlank(query.getParentIds())) {
            String parentIds = StringUtils.appendIfMissing(query.getParentIds(), "%");
            predicates.add(builder.like(root.get("parentIds"), parentIds));
        }
        if (query.getStatus() != null) {
            predicates.add(builder.equal(root.<Integer>get("status"), query.getStatus()));
        }
        if (query.getGradingRoleId() != null) {
            predicates.add(builder.equal(root.<Long>get("gradingRoleId"), query.getGradingRoleId()));
        }
        if (StringUtils.isNotBlank(query.getCompanyCode()) || StringUtils.isNotBlank(query.getCompanyName()) || StringUtils.isNotBlank(query.getCompanyNameEqual()) || StringUtils.isNotBlank(query.getTaxNum())) {
            if (joinCompany == null) {
                joinCompany = root.join("company", JoinType.LEFT);
            }
            if (StringUtils.isNotBlank(query.getCompanyCode())) {
                Set<String> companyCodes = Arrays.stream(StringUtils.split(query.getCompanyCode(), ","))
                        .filter(StringUtils::isNotBlank).collect(Collectors.toSet());
                if(companyCodes != null && !companyCodes.isEmpty()) {
                    if(companyCodes.size() == 1) {
                        predicates.add(builder.equal(joinCompany.<String>get("companyCode"), companyCodes.stream().findFirst().get()));
                    } else {
                        predicates.add(joinCompany.<String>get("companyCode").in(companyCodes));
                    }
                }
            }

            if (StringUtils.isNotBlank(query.getCompanyNameEqual())) {
                predicates.add(builder.equal(joinCompany.<String>get("companyName"), query.getCompanyNameEqual()));
            } else if (StringUtils.isNotBlank(query.getCompanyName())) {
                String companyName = StringUtils.appendIfMissing(query.getCompanyName(), "%");
                predicates.add(builder.like(joinCompany.get("companyName"), companyName));
            }
            if (StringUtils.isNotBlank(query.getTaxNum())) {
                Set<String> taxNums = Arrays.stream(StringUtils.split(query.getTaxNum(), ","))
                        .filter(StringUtils::isNotBlank).collect(Collectors.toSet());
                if(taxNums != null && !taxNums.isEmpty()) {
                    if(taxNums.size() == 1) {
                        predicates.add(builder.equal(joinCompany.<String>get("taxNum"), taxNums.stream().findFirst().get()));
                    } else {
                        predicates.add(joinCompany.<String>get("taxNum").in(taxNums));
                    }
                }
            }
            if (query.getStatus() != null && query.getStatus() == 1) {
                predicates.add(builder.equal(joinCompany.get("status"), 1));
            }
        }
        if (query.getRootOrg() != null && query.getRootOrg()) {
            predicates.add(builder.or(builder.isNull(root.<Long>get("parentId")), builder.equal(root.<Long>get("parentId"), 0)));
        } else {
            if (query.getParentId() != null && query.getParentId() > 0) {
                predicates.add(builder.equal(root.<Long>get("parentId"), query.getParentId()));
            }
        }
        if (!CollectionUtils.isEmpty(query.getIds())) {
            if (query.getIds().size() == 1) {
                predicates.add(builder.equal(root.<Long>get("orgId"), query.getIds().stream().findFirst().get()));
            } else {
                predicates.add(root.<Long>get("orgId").in(query.getIds()));
            }
        }
        //region 如果没有过滤条件则默认 status = 1
        if (predicates.isEmpty()) {
            predicates.add(builder.equal(root.<Integer>get("status"), 1));
        }
        //endregion
        criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
        if (joinTable) {
            if (isCount) {
                criteriaQuery.distinct(true);
            } else {
                criteriaQuery.groupBy(root.<Long>get("orgId"));
            }
        }
        Predicate predicate = criteriaQuery.getRestriction();
        return predicate;
    }


    /**
     * Tuple to Org Mapper
     *
     * @param query
     * @return OrgStruct
     */
    public static Function<Tuple, OrgStruct> tupleMapper(Query query) {
        return new Function<Tuple, OrgStruct>() {
            @Override
            public OrgStruct apply(Tuple tuple) {
                OrgStruct org = tuple.get("org", OrgStruct.class);
                if (org != null) {
                    Company company = tuple.get("company", Company.class);
                    if (company != null) {
                        org.setCompany(company);
                    }
                    Tenant tenant = tuple.get("tenant", Tenant.class);
                    if (tenant != null) {
                        org.setTenant(tenant);
                    }
                    if (query.getWithIsHost() != null && query.getWithIsHost()) {
                        Boolean thisIsHost = tuple.get("thisIsHost", Boolean.class);
                        if (thisIsHost != null) {
                            org.setThisIsHost(thisIsHost);
                        } else {
                            org.setThisIsHost(false);
                        }
                    }
                    if (query.getUserId() != null && query.getUserId() > 0 && query.getWithUserBoundFlag() != null && query.getWithUserBoundFlag()) {
                        Long userBoundCount = tuple.get("userBound", Long.class);
                        if (userBoundCount != null && userBoundCount > 0) {
                            org.setUserBound(true);
                        } else {
                            org.setUserBound(false);
                        }
                    }
                }
                return org;
            }
        };
    }

    public static Specification<OrgStruct> queryOneSpecification(Query query) {
        Specification<OrgStruct> specification = (Specification<OrgStruct>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (query.getOrgId() != null && query.getOrgId() > 0) {
                predicates.add(builder.equal(root.<Long>get("orgId"), query.getOrgId()));
            }
            if (query.getTenantId() != null && query.getTenantId() > 0) {
                predicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
                if (StringUtils.isNotBlank(query.getOrgCode()) || StringUtils.isNotBlank(query.getOrgName()) || (query.getCompanyId() != null && query.getCompanyId() > 0)) {

                    if (StringUtils.isNotBlank(query.getOrgCode())) {
                        predicates.add(builder.equal(root.<String>get("orgCode"), query.getOrgCode()));
                    }
                    if (StringUtils.isNotBlank(query.getOrgName())) {
                        predicates.add(builder.equal(root.<String>get("orgName"), query.getOrgName()));
                    }
                    if (query.getCompanyId() != null && query.getCompanyId() > 0) {
                        predicates.add(builder.equal(root.<Long>get("companyId"), query.getCompanyId()));
                    }
                }
            }

            if (predicates.isEmpty()) {
                throw new IllegalArgumentException("缺少查询参数");
            } else {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return specification;
    }
}
