package com.xforceplus.query;

import com.xforceplus.entity.*;
import com.xforceplus.api.model.RoleModel.Request.Query;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.persistence.Tuple;
import javax.persistence.criteria.*;

import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.Specification;

/**
 * @author geewit
 */
@SuppressWarnings("all")
public class RoleQueryHelper {
    public static Specification<Role> querySpecification(Query query) {
        Specification<Role> specification = (Specification<Role>) (root, criteriaQuery, builder) -> {
            return toPredicate(query, root, criteriaQuery, builder);
        };
        return specification;
    }

    public static Predicate queryTuplePredicate(Query query, Root<Role> root, CriteriaQuery<Tuple> criteriaQuery, CriteriaBuilder builder) {
        return toPredicate(query, root, criteriaQuery, builder);
    }

    public static Predicate queryCountPredicate(Query query, Root<Role> root, CriteriaQuery<Long> criteriaQuery, CriteriaBuilder builder) {
        return toPredicate(query, root, criteriaQuery, builder);
    }

    private static <T> Predicate toPredicate(Query query, Root<Role> root, CriteriaQuery<T> criteriaQuery, CriteriaBuilder builder) {
        List<Predicate> predicates = new ArrayList<>();
        boolean joinTable = false;
        if (query.getType() != null) {
            predicates.add(builder.equal(root.<Integer>get("type"), query.getType()));
        }
        if (query.getId() != null && query.getId() > 0) {
            predicates.add(builder.equal(root.<Long>get("id"), query.getId()));
        } else {
            if (query.getExcludeBoundCurrent() != null && query.getTenantId() != null && query.getTenantId() > 0 && query.getExcludeBoundCurrent() && query.getUserId() != null && query.getUserId() > 0) {
                Subquery<RoleUserRel> subquery = criteriaQuery.subquery(RoleUserRel.class);
                Root roleUserRel = subquery.from(RoleUserRel.class);
                subquery.select(roleUserRel);
                subquery.where(Stream.of(
                        builder.equal(roleUserRel.<Long>get("userId"), query.getUserId()),
                        builder.equal(root.<Long>get("id"), roleUserRel.<Long>get("roleId"))
                ).toArray(Predicate[]::new));
                predicates.add(builder.not(builder.exists(subquery)));
            } else if ((query.getUserId() != null && query.getUserId() > 0) || (query.getAccountId() != null && query.getAccountId() > 0)) {
                ListJoin<Role, RoleUserRel> joinRoleUserRels = root.joinList("roleUserRels", JoinType.LEFT);
                if (query.getUserId() != null && query.getUserId() > 0) {
                    predicates.add(builder.equal(joinRoleUserRels.<Long>get("userId"), query.getUserId()));
                }
                if (query.getAccountId() != null && query.getAccountId() > 0) {
                    Join<RoleUserRel, User> joinUser = joinRoleUserRels.join("user", JoinType.LEFT);
                    predicates.add(builder.equal(joinUser.<Long>get("accountId"), query.getAccountId()));
                }
                joinTable = true;
            }

            if (query.getOrgId() != null && query.getOrgId() > 0) {
                if (query.getType() != null) {
                    if (query.getType() == 1 || query.getType() == 2) {
                        ListJoin<Role, OrgStruct> joinOrgs;
                        if (query.getType() == 1) {
                            joinOrgs = root.joinList("gradingOrgs", JoinType.LEFT);
                        } else {
                            joinOrgs = root.joinList("defaultOrgs", JoinType.LEFT);
                        }
                        predicates.add(builder.equal(joinOrgs.<Long>get("orgId"), query.getOrgId()));
                        joinTable = true;
                    }
                }
            }

            // 是否过滤已过期的, 默认不过滤, 如过滤需提前设置
            if(query.isFilterExpired()) {
                Path<Date> expireDate = root.<Date>get("expireDate");
                predicates.add(builder.or(builder.isNull(expireDate), builder.greaterThan(expireDate, builder.currentDate())));
            }

            if (StringUtils.isNotBlank(query.getRoleCode())) {
                Set<String> codes = Arrays.stream(StringUtils.split(query.getRoleCode(), ","))
                        .filter(StringUtils::isNotBlank).collect(Collectors.toSet());
                if (!codes.isEmpty()) {
                    if(codes.size() == 1) {
                        predicates.add(builder.equal(root.<String>get("code"), codes.stream().findFirst().get()));
                    } else {
                        predicates.add(root.<String>get("code").in(codes));
                    }
                }
            }
            if (StringUtils.isNotBlank(query.getSearchKey())) {
                predicates.add(builder.or(
                        builder.like(root.get("name"), query.getSearchKey() + "%"),
                        builder.equal(root.<String>get("code"), query.getSearchKey())
                ));
            }
            if (StringUtils.isNotBlank(query.getRoleName())) {
                predicates.add(builder.like(root.<String>get("name"), query.getRoleName() + "%"));
            }
            if (query.getFromRoleId() != null && query.getFromRoleId() > 0) {
                predicates.add(builder.equal(root.<Long>get("fromRoleId"), query.getFromRoleId()));
            }
            if (query.getStatus() != null) {
                predicates.add(builder.equal(root.<Integer>get("status"), query.getStatus()));
            }
        }
        Boolean isSystem = query.getIsSystem();
        if (isSystem != null && isSystem) {
            predicates.add(builder.lessThanOrEqualTo(root.get("tenantId"), 0L));
        } else if (query.getTenantId() != null && query.getTenantId() > 0) {
            predicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
        }
        if (!predicates.isEmpty()) {
            criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
        }
        if (joinTable) {
            if (criteriaQuery.getResultType().isAssignableFrom(Long.class)) {
                criteriaQuery.distinct(true);
                return criteriaQuery.getRestriction();
            } else {
                criteriaQuery.groupBy(root.<Long>get("id"));
                return criteriaQuery.getGroupRestriction();
            }
        } else {
            return criteriaQuery.getRestriction();
        }
    }

    public static Specification<Role> queryOneSpecification(Query query) {
        Specification<Role> specification = (Specification<Role>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (query.getId() != null && query.getId() > 0) {
                predicates.add(builder.equal(root.<Long>get("id"), query.getId()));
            } else {
                if (StringUtils.isNotBlank(query.getRoleCode())) {
                    predicates.add(builder.equal(root.<String>get("code"), query.getRoleCode()));
                }
                if (StringUtils.isNotBlank(query.getRoleName())) {
                    predicates.add(builder.equal(root.<String>get("name"), query.getRoleName()));
                }
            }
            Boolean isSystem = query.getIsSystem();
            if (isSystem != null && isSystem) {
                predicates.add(builder.lessThanOrEqualTo(root.get("tenantId"), 0L));
            } else if (query.getTenantId() != null && query.getTenantId() > 0) {
                predicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
            }
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return specification;
    }

    public static Specification<RoleUserRel> queryRoleUserRelSpecification(Long tenantId, Long roleId, Long userId) {
        Specification<RoleUserRel> specification = (Specification<RoleUserRel>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (tenantId != null && tenantId > 0) {
                predicates.add(builder.equal(root.<Long>get("tenantId"), tenantId));
            }
            if (roleId != null && roleId > 0) {
                predicates.add(builder.equal(root.<Long>get("roleId"), roleId));
            }
            if (userId != null && userId > 0) {
                predicates.add(builder.equal(root.<Long>get("userId"), userId));
            }
            // 过滤已过期的
            predicates.add(builder.or(builder.isNull(root.<Date>get("expireDate")), builder.greaterThan(root.<Date>get("expireDate"), new Date())));
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return specification;
    }
}
