package com.xforceplus.query;

import com.xforceplus.api.model.UserModel.Request.Query;
import com.xforceplus.entity.*;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.util.CollectionUtils;

import javax.persistence.Tuple;
import javax.persistence.criteria.*;
import java.util.*;

/**
 * @author geewit
 */
@SuppressWarnings("all")
public class UserQueryHelper {
    private final static Logger logger = LoggerFactory.getLogger(UserQueryHelper.class);

    /**
     * 构建查询 Specification 对象
     *
     * @param query UserDto
     * @return Specification
     */
    public static Specification<User> querySpecification(Query query) {
        Specification<User> specification = (Specification<User>) (root, criteriaQuery, builder) -> {
            return toPredicate(query, root, criteriaQuery, builder);
        };
        return specification;
    }

    public static Predicate queryTuplePredicate(Query query, Root<User> root, CriteriaQuery<Tuple> criteriaQuery, CriteriaBuilder builder) {
        return toPredicate(query, root, criteriaQuery, builder);
    }

    public static Predicate queryCountPredicate(Query query, Root<User> root, CriteriaQuery<Long> criteriaQuery, CriteriaBuilder builder) {
        return toPredicate(query, root, criteriaQuery, builder);
    }

    private static <T> Predicate toPredicate(Query query, Root<User> root, CriteriaQuery<T> criteriaQuery, CriteriaBuilder builder) {
        List<Predicate> filterPredicates = new ArrayList<>();
        List<Predicate> rangePredicates = new ArrayList<>();
        boolean joinTable = false;
        if(query.getOrgVirtualNodeId() != null && query.getOrgVirtualNodeId() > 0) {
            if (query.getExcludeBoundOrgVirtualNode() != null && query.getExcludeBoundOrgVirtualNode() && query.getTenantId() != null && query.getTenantId() > 0) {
                Subquery<OrgVirtualNodeUserRel> subquery = criteriaQuery.subquery(OrgVirtualNodeUserRel.class);
                Root relOrgVirtualTreeRoot = subquery.from(OrgVirtualNodeUserRel.class);
                subquery.select(relOrgVirtualTreeRoot);
                Predicate[] subPredicates = new Predicate[] {
                        builder.equal(relOrgVirtualTreeRoot.<Long>get("orgVirtualNodeId"), query.getOrgVirtualNodeId()),
                        builder.equal(root.<Long>get("id"), relOrgVirtualTreeRoot.<Long>get("userId"))
                };
                subquery.where(subPredicates);
                rangePredicates.add(builder.not(builder.exists(subquery)));
            } else {
                ListJoin<User, OrgVirtualNodeUserRel> userJoinRoleOrgVirtualTreeRel = root.joinList("orgVirtualNodeUserRels", JoinType.LEFT);
                rangePredicates.add(builder.equal(userJoinRoleOrgVirtualTreeRel.<Long>get("orgVirtualNodeId"), query.getOrgVirtualNodeId()));
            }
        }

        if (query.getExcludeBoundCurrent() != null && query.getTenantId() != null && query.getTenantId() > 0 && query.getExcludeBoundCurrent() && query.getRoleId() != null && query.getRoleId() > 0) {
            Subquery<RoleUserRel> subquery = criteriaQuery.subquery(RoleUserRel.class);
            Root<RoleUserRel> roleUserRel = subquery.from(RoleUserRel.class);
            subquery.select(roleUserRel);
            Predicate[] subPredicates = new Predicate[] {
                    builder.equal(roleUserRel.<Long>get("roleId"), query.getRoleId()),
                    builder.equal(root.<Long>get("id"), roleUserRel.<Long>get("userId"))
            };
            subquery.where(subPredicates);
            rangePredicates.add(builder.not(builder.exists(subquery)));
        } else if ((query.getRoleId() != null && query.getRoleId() > 0) || StringUtils.isNotBlank(query.getRoleCode()) || StringUtils.isNotBlank(query.getRoleIds())) {
            Set<Long> roleIds = new HashSet<>();
            if (query.getRoleId() != null && query.getRoleId() > 0) {
                roleIds.add(query.getRoleId());
            }
            if (StringUtils.isNotBlank(query.getRoleIds())) {
                Arrays.stream(StringUtils.split(query.getRoleIds(), ",")).forEach(id -> {
                    try {
                        Long parseLong = Long.parseLong(id);
                        roleIds.add(parseLong);
                    } catch (NumberFormatException e) {
                        String message = e.getMessage() + ", id: " + id;
                        logger.warn(message);
                    }
                });
            }
            ListJoin<User, RoleUserRel> userJoinRoleUserRel = null;
            if (!roleIds.isEmpty()) {
                userJoinRoleUserRel = root.joinList("roleUserRels", JoinType.LEFT);
                if (roleIds.size() == 1) {
                    rangePredicates.add(builder.equal(userJoinRoleUserRel.<Long>get("roleId"), roleIds.stream().findFirst().get()));
                } else {
                    rangePredicates.add(userJoinRoleUserRel.<Long>get("roleId").in(roleIds));
                }
            }
            if (StringUtils.isNotBlank(query.getRoleCode())) {
                if (userJoinRoleUserRel == null) {
                    userJoinRoleUserRel = root.joinList("roleUserRels", JoinType.LEFT);
                }
                Join<RoleUserRel, Role> joinRole = userJoinRoleUserRel.join("role", JoinType.LEFT);
                rangePredicates.add(builder.equal(joinRole.<String>get("code"), query.getRoleCode()));
                if (query.getStatus() != null && query.getStatus() == 1) {
                    rangePredicates.add(builder.equal(joinRole.get("status"), 1));
                }
            }
            joinTable = true;
        }
        if (StringUtils.isNotBlank(query.getAccountName()) || StringUtils.isNotBlank(query.getAccountOptions())) {
            Join<User, Account> joinAccount = root.join("account", JoinType.LEFT);
            if (StringUtils.isNotBlank(query.getAccountName())) {
                rangePredicates.add(builder.or(
                        builder.equal(joinAccount.<String>get("username"), query.getAccountName()),
                        builder.equal(joinAccount.<String>get("telPhone"), query.getAccountName()),
                        builder.equal(joinAccount.<String>get("email"), query.getAccountName())
                ));
            } else if (StringUtils.isNotBlank(query.getAccountOptions())) {
                rangePredicates.add(builder.or(
                        builder.like(joinAccount.get("username"), "%" + query.getAccountOptions() + "%"),
                        builder.like(joinAccount.get("telPhone"), "%" + query.getAccountOptions() + "%"),
                        builder.like(joinAccount.get("email"), "%" + query.getAccountOptions() + "%")
                ));
            }

            if (query.getStatus() != null && query.getStatus() == 1) {
                rangePredicates.add(builder.equal(joinAccount.get("status"), 1));
            }
            joinTable = true;
        }

        if (query.getUserId() != null && query.getUserId() > 0) {
            filterPredicates.add(builder.equal(root.<Long>get("id"), query.getUserId()));
        } else if (!CollectionUtils.isEmpty(query.getUserIds())) {
            if (query.getUserIds().size() == 1) {
                filterPredicates.add(builder.equal(root.<Long>get("id"), query.getUserIds().stream().findFirst().get()));
            } else {
                filterPredicates.add(root.<Long>get("id").in(query.getUserIds()));
            }
        }
        if (StringUtils.isNotBlank(query.getUserCode())) {
            filterPredicates.add(builder.equal(root.<String>get("userCode"), query.getUserCode()));
        }
        if (StringUtils.isNotBlank(query.getUserName())) {
            filterPredicates.add(builder.equal(root.<String>get("userName"), query.getUserName()));
        }
        if (StringUtils.isNotBlank(query.getUserNumber())) {
            filterPredicates.add(builder.equal(root.<String>get("userNumber"), query.getUserNumber()));
        }
        if (StringUtils.isNotBlank(query.getUserPhone())) {
            filterPredicates.add(builder.equal(root.<String>get("userPhone"), query.getUserPhone()));
        }
        if (StringUtils.isNotBlank(query.getUserEmailAddr())) {
            filterPredicates.add(builder.equal(root.<String>get("userEmailAddr"), query.getUserEmailAddr()));
        }
        if (StringUtils.isNotBlank(query.getUserOptions())) {
            filterPredicates.add(builder.or(
                    builder.like(root.get("userName"), query.getUserOptions() + "%"),
                    builder.equal(root.<String>get("userCode"), query.getUserOptions()),
                    builder.equal(root.<String>get("userNumber"), query.getUserOptions()),
                    builder.equal(root.<String>get("userPhone"), query.getUserOptions()),
                    builder.equal(root.<String>get("userEmailAddr"), query.getUserOptions())
            ));
        }
        if (StringUtils.isNotBlank(query.getTenantCode()) || StringUtils.isNotBlank(query.getTenantName())) {
            Join<User, Tenant> joinTenant = root.join("tenant", JoinType.LEFT);
            if (StringUtils.isNotBlank(query.getTenantCode())) {
                rangePredicates.add(builder.equal(joinTenant.<String>get("tenantCode"), query.getTenantCode()));
            }
            if (StringUtils.isNotBlank(query.getTenantName())) {
                rangePredicates.add(builder.like(joinTenant.get("tenantName"), query.getTenantName() + "%"));
            }
        }
        if (query.getStatus() != null) {
            filterPredicates.add(builder.equal(root.<Integer>get("status"), query.getStatus()));
        }
        if (query.getExpired() != null) {
            if (query.getExpired()) {
                filterPredicates.add(builder.lessThan(root.<Date>get("expiredDate"), builder.currentDate()));
            } else {
                filterPredicates.add(builder.greaterThanOrEqualTo(root.<Date>get("expiredDate"), builder.currentDate()));
            }
        }
        if (query.getAfterExpiredDate() != null) {
            filterPredicates.add(builder.lessThan(root.<Date>get("expiredDate"), query.getAfterExpiredDate()));
        }
        if (query.getBeforeExpiredDate() != null) {
            filterPredicates.add(builder.greaterThanOrEqualTo(root.<Date>get("expiredDate"), query.getBeforeExpiredDate()));
        }
        if (query.getAccountId() != null && query.getAccountId() > 0) {
            filterPredicates.add(builder.equal(root.<Long>get("accountId"), query.getAccountId()));
        } else if(!CollectionUtils.isEmpty(query.getAccountIds())) {
            if(query.getAccountIds().size() == 1) {
                filterPredicates.add(builder.equal(root.<Long>get("accountId"), query.getAccountIds().stream().findFirst().get()));
            } else {
                filterPredicates.add(root.<Long>get("accountId").in(query.getAccountIds()));
            }
        }

        if (!CollectionUtils.isEmpty(query.getFilterOrgParentIds()) || !CollectionUtils.isEmpty(query.getOrgIds())) {
            ListJoin<User, OrgUserRel> joinOrgRels = root.joinList("orgUserRels", JoinType.LEFT);
            if (!CollectionUtils.isEmpty(query.getOrgIds())) {
                if (query.getOrgIds().size() == 1) {
                    rangePredicates.add(builder.equal(joinOrgRels.<Long>get("orgStructId"), query.getOrgIds().stream().findFirst().get()));
                } else {
                    rangePredicates.add(joinOrgRels.<Long>get("orgStructId").in(query.getOrgIds()));
                }
            }
            if (!CollectionUtils.isEmpty(query.getFilterOrgParentIds())) {
                List<String> filterParentIdsList = new ArrayList<>(query.getFilterOrgParentIds());
                Collections.sort(filterParentIdsList);
                Set<String> filterParentIdsSet = new HashSet<>();
                Join<OrgUserRel, OrgStruct> joinOrg = joinOrgRels.join("org", JoinType.LEFT);
                for (final String filterParentIds : filterParentIdsList) {
                    if (filterParentIdsSet.stream().noneMatch(parentIds -> StringUtils.startsWith(filterParentIds, parentIds))) {
                        filterParentIdsSet.add(filterParentIds);
                        rangePredicates.add(builder.like(joinOrg.<String>get("parentIds"), StringUtils.appendIfMissing(filterParentIds, "%")));
                    }
                }
            }
            joinTable = true;
        }

        Predicate orPredicate = null;
        if (query.getTenantId() != null && query.getTenantId() > 0) {
            rangePredicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
            if (query.getIncludeIndependents() != null && query.getIncludeIndependents()) {
                Subquery<Long> subquery = criteriaQuery.subquery(Long.class);
                Root<OrgUserRel> orgUserRel = subquery.from(OrgUserRel.class);
                subquery.select(builder.treat(orgUserRel.<Long>get("userId"), Long.class));
                subquery.groupBy(orgUserRel.<Long>get("userId"));
                Predicate subPredicate = builder.equal(orgUserRel.<Long>get("tenantId"), query.getTenantId());
                subquery.where(subPredicate);
                orPredicate = builder.not((root.<Long>get("id").in(subquery)));
            }
        }

        if (!rangePredicates.isEmpty() || !filterPredicates.isEmpty()) {
            if (orPredicate == null) {
                rangePredicates.addAll(filterPredicates);
                criteriaQuery.where(rangePredicates.stream().toArray(Predicate[]::new));
            } else {
                Predicate rangePredicate = builder.and(rangePredicates.stream().toArray(Predicate[]::new));
                Predicate filterPredicate = builder.and(filterPredicates.stream().toArray(Predicate[]::new));
                criteriaQuery.where(builder.and(builder.or(rangePredicate, orPredicate), filterPredicate));
            }
        } else if (orPredicate != null) {
            criteriaQuery.where(orPredicate);
        }
        if (joinTable) {
            if (criteriaQuery.getResultType().isAssignableFrom(Long.class)) {
                criteriaQuery.distinct(true);
                return criteriaQuery.getRestriction();
            } else {
                criteriaQuery.groupBy(root.<Long>get("id"));
                return criteriaQuery.getGroupRestriction();
            }
        } else {
            return criteriaQuery.getRestriction();
        }
    }

    /**
     * 构建查询Specification对象
     *
     * @param query UserDto
     * @return Specification
     */
    public static Specification<User> queryOneSpecification(Query query) {
        Specification<User> specification = (Specification<User>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (query.getUserId() != null && query.getUserId() > 0) {
                predicates.add(builder.equal(root.<Long>get("id"), query.getUserId()));
            }
            if (StringUtils.isNotBlank(query.getUserCode())) {
                predicates.add(builder.equal(root.<String>get("userCode"), query.getUserCode()));
            }
            if (query.getTenantId() != null && query.getTenantId() > 0) {
                predicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
            }
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return specification;
    }
}
