package com.xforceplus.entity;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.xforceplus.domain.company.CompanyTenantRelDto;
import com.xforceplus.domain.tenant.TenantDto;
import com.xforceplus.jpa.listener.CompanyTenantRelListener;
import io.geewit.data.jpa.essential.id.IDGenerators;
import lombok.Setter;
import org.hibernate.annotations.DynamicInsert;
import org.hibernate.annotations.DynamicUpdate;
import org.hibernate.annotations.GenericGenerator;

import javax.persistence.*;
import java.util.Date;
import java.util.List;

/**
 * 公司租户关联关系 Entity
 *
 * @author geewit
 */
@Setter
@JsonIgnoreProperties(value = {"hibernateLazyInitializer", "handler", "fieldHandler"}, ignoreUnknown = true)
@NamedEntityGraphs({
        @NamedEntityGraph(name = CompanyTenantRel.NAMED_ENTITY_GRAPH_DEFAULT,
                attributeNodes = {
                        @NamedAttributeNode("company"),
                        @NamedAttributeNode("tenant"),
                        @NamedAttributeNode("relatedCompany"),
                        @NamedAttributeNode("relatedTenant")
                }
        )})
@EntityListeners({CompanyTenantRelListener.class})
@DynamicInsert
@DynamicUpdate
@Entity
@Table(name = "company_tenant_rel")
public class CompanyTenantRel extends CompanyTenantRelDto {
    public static final String NAMED_ENTITY_GRAPH_DEFAULT = "CompanyTenantRel.graph";


    /**
     * 关联的租户
     */
    @JsonIgnore
    private Tenant tenant;
    /**
     * 关联的公司
     */
    @JsonIgnore
    private Company company;
    /**
     * 被关联的租户
     */
    @JsonIgnore
    private Tenant relatedTenant;
    /**
     * 被关联的公司
     */
    @JsonIgnore
    private Company relatedCompany;
    /**
     * 关联的组织
     */
    @JsonIgnore
    private List<OrgStruct> orgs;
    /**
     * 被关联的组织
     */
    @JsonIgnore
    private List<OrgStruct> relatedOrgs;

    @Override
    @Id
    @Column(name = "id", nullable = false)
    @GeneratedValue(strategy = GenerationType.TABLE, generator = "SnowflakeGenerator")
    @GenericGenerator(name = "SnowflakeGenerator", strategy = IDGenerators.SnowflakeGenerator)
    public Long getId() {
        return id;
    }

    /**
     * 邀请方公司id
     * @return 邀请方公司id
     */
    @Override
    @Basic
    @Column(name = "inviting_company_id", updatable = false)
    public Long getInvitingCompanyId() {
        return invitingCompanyId;
    }

    /**
     * 邀请方公司名称
     * @return 邀请方公司名称
     */
    @Override
    @Basic
    @Column(name = "inviting_company_name", updatable = false)
    public String getInvitingCompanyName() {
        return invitingCompanyName;
    }

    @Override
    @Basic
    @Column(name = "tenant_id", updatable = false)
    public Long getTenantId() {
        return tenantId;
    }

    @Override
    @Basic
    @Column(name = "company_id", updatable = false)
    public Long getCompanyId() {
        return companyId;
    }

    @Override
    @Basic
    @Column(name = "related_tenant_id", updatable = false)
    public Long getRelatedTenantId() {
        return relatedTenantId;
    }

    @Override
    @Basic
    @Column(name = "related_company_id", updatable = false)
    public Long getRelatedCompanyId() {
        return relatedCompanyId;
    }

    @Override
    @Basic
    @Column(name = "switches")
    public Integer getSwitches() {
        return switches;
    }

    @Override
    @Temporal(TemporalType.DATE)
    @Column(name = "statement_start_date")
    public Date getStatementStartDate() {
        return statementStartDate;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "statement_granted_time")
    public Date getStatementGrantedTime() {
        return statementGrantedTime;
    }

    @Override
    @Temporal(TemporalType.DATE)
    @Column(name = "invoice_start_date")
    public Date getInvoiceStartDate() {
        return invoiceStartDate;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "invoice_granted_time")
    public Date getInvoiceGrantedTime() {
        return invoiceGrantedTime;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "create_time", updatable = false)
    public Date getCreateTime() {
        return createTime;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "update_time")
    public Date getUpdateTime() {
        return updateTime;
    }

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "tenant_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Tenant getTenant() {
        return tenant;
    }

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "company_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Company getCompany() {
        return company;
    }

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "related_tenant_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Tenant getRelatedTenant() {
        return relatedTenant;
    }

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "related_company_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Company getRelatedCompany() {
        return relatedCompany;
    }

    @OneToMany(fetch = FetchType.LAZY)
    @JoinColumns(value = {
            @JoinColumn(name = "tenant_id", referencedColumnName = "tenant_id", insertable = false, updatable = false),
            @JoinColumn(name = "company_id", referencedColumnName = "company_id", insertable = false, updatable = false)
    }, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public List<OrgStruct> getOrgs() {
        return orgs;
    }

    @OneToMany(fetch = FetchType.LAZY)
    @JoinColumns(value = {
            @JoinColumn(name = "related_tenant_id", referencedColumnName = "tenant_id", insertable = false, updatable = false),
            @JoinColumn(name = "company_id", referencedColumnName = "company_id", insertable = false, updatable = false)
    }, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public List<OrgStruct> getRelatedOrgs() {
        return relatedOrgs;
    }

    @JsonIgnore
    @Transient
    public void fillTenant() {
        if(this.tenant != null) {
            super.tenantCode = this.tenant.getTenantCode();
            super.tenantName = this.tenant.getTenantName();
        } else {
            super.tenantId = null;
        }
    }

    @JsonIgnore
    @Transient
    public void fillHostTenant(TenantDto hostTenant) {
        if(hostTenant != null) {
            super.hostTenantName = hostTenant.getTenantName();
            super.hostTenantCode = hostTenant.getTenantCode();
        } else {
            super.hostTenantId = null;
        }
    }

    @JsonIgnore
    @Transient
    public void fillCompany() {
        if(company != null) {
            super.companyCode = company.getCompanyCode();
            super.companyName = company.getCompanyName();
            super.taxNum = company.getTaxNum();
        }
    }

    @JsonIgnore
    @Transient
    public void fillRelatedEntity() {
        if(relatedTenant != null) {
            super.relatedTenantName = relatedTenant.getTenantName();
            super.relatedTenantCode = relatedTenant.getTenantCode();
        } else {
            super.relatedTenantId = null;
        }
        if(relatedCompany != null) {
            super.relatedCompanyCode = relatedCompany.getCompanyCode();
            super.relatedCompanyName = relatedCompany.getCompanyName();
            super.relatedTaxNum = relatedCompany.getTaxNum();
        } else {
            super.relatedCompanyId = null;
        }
    }
}
