package com.xforceplus.entity;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.xforceplus.domain.tenant.TenantPolicyDto;
import com.xforceplus.jpa.listener.TenantPolicyListener;
import io.geewit.data.jpa.essential.id.IDGenerators;
import lombok.Setter;
import org.hibernate.annotations.*;

import javax.persistence.Entity;
import javax.persistence.ForeignKey;
import javax.persistence.Table;
import javax.persistence.*;
import java.io.Serializable;
import java.util.Date;
import java.util.Objects;

/**
 * @author geewit
 */
@Setter
@JsonIgnoreProperties(value = {"hibernateLazyInitializer", "handler", "fieldHandler"}, ignoreUnknown = true)
@EntityListeners({TenantPolicyListener.class})
@NamedEntityGraphs({
        @NamedEntityGraph(name = TenantPolicy.NAMED_ENTITY_GRAPH_DEFAULT,
                attributeNodes = {
                        @NamedAttributeNode("tenant")
                }
        )
})
@DynamicInsert
@DynamicUpdate
@Entity
@Table(name = "tenant_policy")
public class TenantPolicy extends TenantPolicyDto implements Serializable {

    public static final String NAMED_ENTITY_GRAPH_DEFAULT = "TenantPolicy.graph";

    public static TenantPolicy ofDefault(String name, String policy) {
        TenantPolicy tenantPolicy = new TenantPolicy();
        tenantPolicy.setName(name);
        tenantPolicy.setPolicy(policy);
        return tenantPolicy;
    }

    @JsonIgnore
    private Tenant tenant;

    @Override
    @Id
    @Column(name = "id", nullable = false)
    @GeneratedValue(strategy = GenerationType.TABLE, generator = "SnowflakeGenerator")
    @GenericGenerator(name = "SnowflakeGenerator", strategy = IDGenerators.SnowflakeGenerator)
    public Long getId() {
        return id;
    }

    @Override
    @Basic
    @Column(name = "tenant_id", nullable = false)
    public Long getTenantId() {
        return tenantId;
    }

    @Override
    @Basic
    @Column(name = "name", nullable = false)
    public String getName() {
        return name;
    }

    @Override
    @Basic
    @Column(name = "policy", nullable = false)
    public String getPolicy() {
        return policy;
    }

    @Override
    @Basic
    @Column(name = "updater_id")
    public String getUpdaterId() {
        return updaterId;
    }

    @Override
    @Basic
    @Column(name = "updater_name")
    public String getUpdaterName() {
        return updaterName;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "update_time")
    public Date getUpdateTime() {
        return updateTime;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "tenant_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Tenant getTenant() {
        return tenant;
    }

    @Transient
    public void postLoad() {
        if (this.tenant != null) {
            super.tenantName = this.tenant.getTenantName();
            super.tenantCode = this.tenant.getTenantCode();
        }
    }

    @Override
    public boolean equals(Object o) {
        if (null == o){
            return false;
        }
        if (this == o) {
            return true;
        }
        if (this.getClass() != o.getClass()) {
            return false;
        }
        TenantPolicy that = (TenantPolicy) o;
        return this.tenantId.equals(that.tenantId) &&
                this.name.equals(that.name);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.tenantId, this.name);
    }
}
