package com.xforceplus.jpa.listener;


import com.xforceplus.entity.Company;
import com.xforceplus.event.dto.CompanyNameChanged;
import com.xforceplus.event.model.EntityPostSaveEvent;
import io.geewit.data.jpa.essential.id.SnowflakeGenerator;
import io.geewit.utils.uuid.UUID;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.PostLoad;
import javax.persistence.PostUpdate;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Objects;


/**
 * 公司自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class CompanyListener extends OperatorListener<Company> implements ApplicationEventPublisherAware {

    private static final Logger logger = LoggerFactory.getLogger(OrgListener.class);

    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    private final ThreadLocal<CompanyNameChanged> companyNameChangedThreadLocal = new ThreadLocal<>();

    @PrePersist
    public void prePersist(Company entity) {
        if (entity.getCompanyId() == null) {
            long id = SnowflakeGenerator.id(Company.class);
            entity.setCompanyId(id);
        }
        if (StringUtils.isEmpty(entity.getCompanyCode())) {
            String companyCode = UUID.randomUUID().toString();
            entity.setCompanyCode(companyCode);
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (StringUtils.isBlank(entity.getTaxNum())) {
            throw new IllegalArgumentException("公司税号不能为空");
        }
        if (StringUtils.isBlank(entity.getCompanyName())) {
            throw new IllegalArgumentException("公司名称不能为空");
        }
        if (entity.getEffective() == null) {
            entity.setEffective(true);
        }
        this.cleanRelatedEntities(entity);
        super.beforeInsert(entity);
        this.trail(RevisionMetadata.RevisionType.INSERT, entity);
    }

    @PreUpdate
    public void preUpdate(Company entity) {
        if (entity.getEffective() == null) {
            entity.setEffective(true);
        }
        boolean newCompanyName = false;
        String preCompanyName = null;
        if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            if (entityEntry != null) {
                preCompanyName = (String) entityEntry.getLoadedValue("companyName");
                newCompanyName = !Objects.equals(preCompanyName, entity.getCompanyName());
            } else {
                newCompanyName = entity.getCompanyName() != null;
            }
        }
        if (newCompanyName) {
            CompanyNameChanged companyNameChanged = CompanyNameChanged.builder()
                    .companyId(entity.getCompanyId())
                    .preCompanyName(preCompanyName)
                    .postCompanyName(entity.getCompanyName())
                    .build();
            companyNameChangedThreadLocal.set(companyNameChanged);
        }
        this.cleanRelatedEntities(entity);
        super.beforeUpdate(entity);
    }

    @PostUpdate
    public void postUpdate(Company entity) {
        CompanyNameChanged companyNameChanged = companyNameChangedThreadLocal.get();
        if (companyNameChanged != null) {
            EntityPostSaveEvent<CompanyNameChanged> entityUpdateEvent = new EntityPostSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, companyNameChanged);
            applicationEventPublisher.publishEvent(entityUpdateEvent);
        }
        this.trail(RevisionMetadata.RevisionType.UPDATE, entity);
        companyNameChangedThreadLocal.remove();
    }

    private void cleanRelatedEntities(Company entity) {
        entity.setTenantRels(null);
        entity.setHostOrgs(null);
        entity.setHostTenant(null);
        entity.setOrgs(null);
    }

    @PostLoad
    public void postLoad(Company entity) {
        entity.postLoad();
    }

    public void trail(RevisionMetadata.RevisionType type, Company entity) {
        EntityPostSaveEvent<Company> entityTrailEvent = new EntityPostSaveEvent<>(type, entity);
        try {
            applicationEventPublisher.publishEvent(entityTrailEvent);
        } catch (Exception e) {
            logger.error("tail publisher error", e);
        }
    }
}