package com.xforceplus.jpa.listener;


import com.xforceplus.entity.ServicePackage;
import com.xforceplus.event.dto.ServicePackageCodeChanged;
import com.xforceplus.event.dto.ServicePackageNameChanged;
import com.xforceplus.event.model.EntityPreSaveEvent;
import io.geewit.utils.uuid.UUID;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Objects;


/**
 * 资源码自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class ServicePackageListener extends OperatorListener<ServicePackage> implements ApplicationEventPublisherAware {
    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    @PrePersist
    public void prePersist(ServicePackage entity) {
        if (StringUtils.isBlank(entity.getServicePackageCode())) {
            entity.setServicePackageCode(UUID.randomUUID().toString());
        }
        //region 发送校验事件
        EntityPreSaveEvent<ServicePackageCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, ServicePackageCodeChanged.builder().entity(entity).build());
        applicationEventPublisher.publishEvent(codeValidationEvent);
        EntityPreSaveEvent<ServicePackageNameChanged> nameValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, ServicePackageNameChanged.builder().entity(entity).build());
        applicationEventPublisher.publishEvent(nameValidationEvent);
        //endregion
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        super.beforeInsert(entity);
        this.cleanRelatedEntities(entity);
    }

    @PreUpdate
    public void preUpdate(ServicePackage entity) {
        if (entity instanceof ManagedEntity) {
            boolean codeChanged;
            boolean appIdChanged;
            boolean nameChanged;
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            if (entityEntry != null) {
                codeChanged = !Objects.equals(entityEntry.getLoadedValue("servicePackageCode"), entity.getServicePackageCode());
                appIdChanged = !Objects.equals(entityEntry.getLoadedValue("appId"), entity.getAppId());
                nameChanged = !Objects.equals(entityEntry.getLoadedValue("servicePackageName"), entity.getServicePackageName());
            } else {
                codeChanged = entity.getServicePackageCode() != null;
                appIdChanged = entity.getAppId() != null;
                nameChanged = entity.getServicePackageName() != null;
            }

            if (codeChanged) {
                //region 发送校验事件
                EntityPreSaveEvent<ServicePackageCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, ServicePackageCodeChanged.builder().entity(entity).build());
                applicationEventPublisher.publishEvent(codeValidationEvent);
                //endregion
            }

            if (appIdChanged || nameChanged) {
                //region 发送校验事件
                EntityPreSaveEvent<ServicePackageNameChanged> nameValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, ServicePackageNameChanged.builder().entity(entity).build());
                applicationEventPublisher.publishEvent(nameValidationEvent);
                //endregion
            }
        }
        super.beforeUpdate(entity);
        this.cleanRelatedEntities(entity);
    }

    private void cleanRelatedEntities(ServicePackage entity) {
        entity.setServiceResourcesetRels(null);
        entity.setCompanyServiceRels(null);
        entity.setApp(null);
    }

    @PostLoad
    public void postLoad(ServicePackage entity) {
        entity.postLoad();
    }
}