package com.xforceplus.jpa.listener;


import com.xforceplus.entity.Tenant;
import com.xforceplus.event.dto.TenantCodeChanged;
import com.xforceplus.event.dto.TenantNameChanged;
import com.xforceplus.event.model.EntityPostSaveEvent;
import com.xforceplus.event.model.EntityPreSaveEvent;
import io.geewit.data.jpa.essential.id.SnowflakeGenerator;
import io.geewit.utils.uuid.UUID;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.PostUpdate;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Objects;


/**
 * 租户自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class TenantListener extends OperatorListener<Tenant> implements ApplicationEventPublisherAware {

    private static final Logger logger = LoggerFactory.getLogger(TenantListener.class);

    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    private final ThreadLocal<TenantNameChanged> tenantNameChangedThreadLocal = new ThreadLocal<>();

    @PrePersist
    public void prePersist(Tenant entity) {
        if (entity.getTenantId() == null) {
            long id = SnowflakeGenerator.id(Tenant.class);
            entity.setTenantId(id);
        }
        if (entity.getTenantCode() == null) {
            entity.setTenantCode(UUID.randomUUID().toString());
        }
        if (entity.getTenantDesc() == null) {
            if (entity.getTenantName() != null) {
                entity.setTenantDesc(entity.getTenantName());
            }
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        //region 发送校验事件
        EntityPreSaveEvent<TenantCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, TenantCodeChanged.builder().entity(entity).build());
        applicationEventPublisher.publishEvent(codeValidationEvent);
        EntityPreSaveEvent<TenantNameChanged> nameValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, TenantNameChanged.builder().entity(entity).build());
        applicationEventPublisher.publishEvent(nameValidationEvent);
        //endregion
        super.beforeInsert(entity);
        this.cleanRelatedEntities(entity);
        this.trail(RevisionMetadata.RevisionType.INSERT, entity);
    }

    @PreUpdate
    public void preUpdate(Tenant entity) {
        if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            boolean codeChanged;
            boolean nameChanged;
            if (entityEntry != null) {
                codeChanged = !Objects.equals(entityEntry.getLoadedValue("tenantCode"), entity.getTenantCode());
                nameChanged = !Objects.equals(entityEntry.getLoadedValue("tenantName"), entity.getTenantName());
            } else {
                codeChanged = entity.getTenantCode() != null;
                nameChanged = entity.getTenantName() != null;
            }

            if (codeChanged) {
                //region 发送校验事件
                EntityPreSaveEvent<TenantCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, TenantCodeChanged.builder().entity(entity).build());
                applicationEventPublisher.publishEvent(codeValidationEvent);
                //endregion
            }
            if (nameChanged) {
                //region 发送校验事件
                TenantNameChanged tenantNameChanged = TenantNameChanged.builder().entity(entity).build();
                EntityPreSaveEvent<TenantNameChanged> nameValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, tenantNameChanged);
                applicationEventPublisher.publishEvent(nameValidationEvent);
                tenantNameChangedThreadLocal.set(tenantNameChanged);
                //endregion
            }
        }
        super.beforeUpdate(entity);
        this.cleanRelatedEntities(entity);
    }

    @PostUpdate
    public void postUpdate(Tenant entity) {
        TenantNameChanged tenantNameChanged = tenantNameChangedThreadLocal.get();
        if (tenantNameChanged != null) {
            //region 发送校验事件
            EntityPostSaveEvent<TenantNameChanged> nameValidationEvent = new EntityPostSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, tenantNameChanged);
            applicationEventPublisher.publishEvent(nameValidationEvent);
            //endregion
        }
        this.trail(RevisionMetadata.RevisionType.UPDATE, entity);
        tenantNameChangedThreadLocal.remove();
    }

    private void cleanRelatedEntities(Tenant entity) {
        entity.setOrgs(null);
        entity.setUsers(null);
    }

    public void trail(RevisionMetadata.RevisionType type, Tenant entity) {
        EntityPostSaveEvent<Tenant> entityTrailEvent = new EntityPostSaveEvent<>(type, entity);
        try {
            applicationEventPublisher.publishEvent(entityTrailEvent);
        } catch (Exception e) {
            logger.error("tail publisher error", e);
        }
    }
}