package com.xforceplus.jpa.listener;


import com.xforceplus.event.model.EntityPostSaveEvent;
import com.xforceplus.event.model.EntityPreSaveEvent;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.PasswordHistory;
import io.geewit.data.jpa.essential.id.SnowflakeGenerator;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.PostPersist;
import javax.persistence.PostUpdate;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Calendar;
import java.util.Date;
import java.util.Objects;


/**
 * 帐号自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class AccountListener extends OperatorListener<Account> implements ApplicationEventPublisherAware {
    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    private final ThreadLocal<Boolean> passwordChangedThreadLocal = new ThreadLocal<>();

    @PrePersist
    public void prePersist(Account entity) {
        //region 发送校验事件
        EntityPreSaveEvent<Account> entityPreSaveEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, entity);
        applicationEventPublisher.publishEvent(entityPreSaveEvent);
        //endregion
        if (entity.getAccountId() == null) {
            long id = SnowflakeGenerator.id(Account.class);
            entity.setAccountId(id);
        }
        if (StringUtils.isBlank(entity.getUsername())) {
            entity.setUsername(StringUtils.EMPTY);
        }
        if (StringUtils.isBlank(entity.getTelPhone())) {
            entity.setTelPhone(StringUtils.EMPTY);
        }
        if (StringUtils.isBlank(entity.getEmail())) {
            entity.setEmail(StringUtils.EMPTY);
        }
        if (entity.getBindAuthFlag() == null) {
            entity.setBindAuthFlag(false);
        }
        if (entity.getChangePasswordFlag() == null) {
            entity.setChangePasswordFlag(false);
        }
        if (entity.getDoubleAuthFlag() == null) {
            entity.setDoubleAuthFlag(false);
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        Date now = Calendar.getInstance().getTime();
        if(null == entity.getUpdateTime()){
            entity.setUpdateTime(now);
        }
        if(null == entity.getCreateTime()){
            entity.setCreateTime(now);
        }
        this.cleanRelatedEntities(entity);
        super.beforeInsert(entity);
    }

    @PreUpdate
    public void preUpdate(Account entity) {
        if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            boolean passwordChanged;
            if (entityEntry != null) {
                passwordChanged = !Objects.equals(entityEntry.getLoadedValue("password"), entity.getPassword());
            } else {
                passwordChanged = entity.getPassword() != null;
            }

            if (passwordChanged) {
                entity.setChangePasswordFlag(false);
                entity.setPwdLastUpdateTime(Calendar.getInstance().getTime());
                //region 发送校验事件
                EntityPreSaveEvent<Account> entityValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, entity);
                applicationEventPublisher.publishEvent(entityValidationEvent);
                //endregion
            }
            passwordChangedThreadLocal.set(passwordChanged);
        }

        if (StringUtils.isBlank(entity.getUsername())) {
            entity.setUsername(StringUtils.EMPTY);
        }
        if (StringUtils.isBlank(entity.getTelPhone())) {
            entity.setTelPhone(StringUtils.EMPTY);
        }
        if (StringUtils.isBlank(entity.getEmail())) {
            entity.setEmail(StringUtils.EMPTY);
        }
        this.cleanRelatedEntities(entity);
        super.beforeUpdate(entity);

    }

    @PostPersist
    public void postPersist(Account entity) {
        this.postSave(RevisionMetadata.RevisionType.INSERT, entity);
    }

    @PostUpdate
    public void postUpdate(Account entity) {
        this.postSave(RevisionMetadata.RevisionType.UPDATE, entity);
    }

    private void postSave(RevisionMetadata.RevisionType revisionType, Account entity) {
        boolean passwordChanged = ObjectUtils.defaultIfNull(passwordChangedThreadLocal.get(), true);
        if (passwordChanged) {
            PasswordHistory passwordHistory = new PasswordHistory();
            passwordHistory.setTenantId(entity.getTenantId());
            passwordHistory.setAccountId(entity.getAccountId());
            passwordHistory.setPassword(entity.getPassword());
            //region 发送保存事件
            EntityPostSaveEvent<PasswordHistory> entityPostSaveEvent = new EntityPostSaveEvent<>(revisionType, passwordHistory);
            applicationEventPublisher.publishEvent(entityPostSaveEvent);
            //endregion
        }
        passwordChangedThreadLocal.remove();
    }

    private void cleanRelatedEntities(Account entity) {
        entity.setUsers(null);
    }
}