package com.xforceplus.jpa.listener;


import com.xforceplus.entity.OrgStruct;
import com.xforceplus.event.dto.OrgCodeChanged;
import com.xforceplus.event.dto.OrgParentIdsBuild;
import com.xforceplus.event.dto.OrgParentIdsRebuild;
import com.xforceplus.event.model.EntityPostSaveEvent;
import com.xforceplus.event.model.EntityPreSaveEvent;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.data.jpa.essential.id.SnowflakeGenerator;
import io.geewit.utils.uuid.UUIDUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.PostLoad;
import javax.persistence.PostUpdate;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Calendar;
import java.util.Objects;


/**
 * 组织自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
@SuppressWarnings("all")
public class OrgListener extends OperatorListener<OrgStruct> implements ApplicationEventPublisherAware {
    private final static Logger logger = LoggerFactory.getLogger(OrgListener.class);

    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    private final ThreadLocal<OrgParentIdsRebuild> orgParentIdsRebuildThreadLocal = new ThreadLocal<>();

    @PrePersist
    public void prePersist(OrgStruct entity) {
        if (entity.getOrgId() == null) {
            long id = SnowflakeGenerator.id(OrgStruct.class);
            entity.setOrgId(id);
        }
        if (StringUtils.isEmpty(entity.getOrgCode())) {
            entity.setOrgCode(UUIDUtils.randomUUID().toUpperCase());
        }
        if (entity.getOrgDesc() == null) {
            entity.setOrgDesc(entity.getOrgName());
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (entity.getAuditStatus() == null) {
            entity.setAuditStatus(1);
        }
        if (entity.getStatus() == 1) {
            entity.setEnabledTime(Calendar.getInstance().getTime());
        }
        this.preSave(RevisionMetadata.RevisionType.INSERT, entity);
        super.beforeInsert(entity);
        this.trail(RevisionMetadata.RevisionType.INSERT, entity);
    }

    @PreUpdate
    public void preUpdate(OrgStruct entity) {
        this.preSave(RevisionMetadata.RevisionType.UPDATE, entity);
        super.beforeUpdate(entity);
    }

    private void preSave(RevisionMetadata.RevisionType revisionType, OrgStruct entity) {
        boolean newOrgCode = false;
        if (RevisionMetadata.RevisionType.INSERT.equals(revisionType)) {
            newOrgCode = true;
        } else if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            if (entityEntry != null) {
                newOrgCode = !Objects.equals(entityEntry.getLoadedValue("orgCode"), entity.getOrgCode());
            } else {
                newOrgCode = entity.getOrgCode() != null;
            }
        }
        if (newOrgCode) {
            OrgCodeChanged orgCodeValidation = OrgCodeChanged.builder()
                    .entity(entity)
                    .build();
            EntityPreSaveEvent<OrgCodeChanged> entityValidationEvent = new EntityPreSaveEvent<>(revisionType, orgCodeValidation);
            applicationEventPublisher.publishEvent(entityValidationEvent);
        }
        if (entity.getCompanyId() != null && entity.getCompanyId() <= 0) {
            entity.setCompanyId(null);
        }
        if (entity.getParentId() == null) {
            entity.setCompanyId(null);
            entity.setOrgType(OrgType.GROUP);
        } else if (entity.getParentId() <= 0) {
            entity.setParentId(null);
        }
        if (entity.getCompanyId() == null) {
            if (OrgType.COMPANY.equals(entity.getOrgType())) {
                if (entity.getParentId() == null) {
                    entity.setOrgType(OrgType.GROUP);
                }
            }
        } else if (entity.getCompanyId() > 0) {
            entity.setOrgType(OrgType.COMPANY);
        }

        if (entity.getOrgType() == null) {
            entity.setCompanyId(null);
            entity.setOrgType(OrgType.NORMAL);
        }
        if (entity.getGradingRoleId() != null && entity.getGradingRoleId() <= 0) {
            entity.setGradingRoleId(null);
        }
        if (entity.getDefaultOrgRoleId() != null && entity.getDefaultOrgRoleId() <= 0) {
            entity.setDefaultOrgRoleId(null);
        }
        boolean newParentId = false;
        boolean newOrgType = false;
        boolean newStatus = false;
        EntityEntry entityEntry = null;
        if (RevisionMetadata.RevisionType.INSERT.equals(revisionType)) {
            newParentId = true;
            newStatus = true;
        } else if (entity instanceof ManagedEntity) {
            entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            if (entityEntry != null) {
                newParentId = !Objects.equals(entityEntry.getLoadedValue("parentId"), entity.getParentId());
                newOrgType = !Objects.equals(entityEntry.getLoadedValue("orgType"), entity.getOrgType());
                newStatus = !Objects.equals(entityEntry.getLoadedValue("status"), entity.getStatus());
            }
        }
        if (newParentId || newOrgType) {
            OrgParentIdsBuild orgParentIdsBuild = OrgParentIdsBuild.builder()
                    .entity(entity)
                    .build();
            EntityPreSaveEvent<OrgParentIdsBuild> entityBuildParentIdsEvent = new EntityPreSaveEvent<>(revisionType, orgParentIdsBuild);
            applicationEventPublisher.publishEvent(entityBuildParentIdsEvent);
            OrgParentIdsRebuild.OrgParentIdsRebuildBuilder rebuildBuilder = OrgParentIdsRebuild.builder();
            if (entityEntry != null) {
                String previousParentIds = (String)entityEntry.getLoadedValue("parentIds");
                rebuildBuilder.previousParentIds(previousParentIds);
            }
            OrgParentIdsRebuild orgParentIdsRebuild = rebuildBuilder.build();
            orgParentIdsRebuildThreadLocal.set(orgParentIdsRebuild);
        }

        if (newStatus) {
            if (entity.getStatus() != null) {
                if (entity.getStatus() == 1 && entity.getEnabledTime() == null) {
                    entity.setEnabledTime(Calendar.getInstance().getTime());
                } else if (entity.getStatus() == 0) {
                    entity.setDisabledTime(Calendar.getInstance().getTime());
                }
            }
        }
        entity.setTenant(null);
        entity.setCompanyNos(null);
        entity.setParent(null);
        entity.setChildren(null);
        entity.setDefaultOrgRole(null);
        entity.setGradingRole(null);
        entity.setOrgUserRels(null);
    }

    @PostLoad
    public void postLoad(OrgStruct entity) {
        entity.postLoad();
    }

    @PostUpdate
    public void postUpdate(OrgStruct entity) {
        OrgParentIdsRebuild orgParentIdsRebuild = orgParentIdsRebuildThreadLocal.get();
        if (orgParentIdsRebuild != null) {
            String nextParentIds = entity.getParentIds();
            orgParentIdsRebuild.setNextParentIds(nextParentIds);
            EntityPostSaveEvent<OrgParentIdsRebuild> rebuildParentIdsEvent = new EntityPostSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, orgParentIdsRebuild);
            applicationEventPublisher.publishEvent(rebuildParentIdsEvent);
        }
        this.trail(RevisionMetadata.RevisionType.UPDATE, entity);
        orgParentIdsRebuildThreadLocal.remove();
    }

    public void trail(RevisionMetadata.RevisionType type, OrgStruct entity) {
        EntityPostSaveEvent<OrgStruct> entityTrailEvent = new EntityPostSaveEvent<>(type, entity);
        try {
            applicationEventPublisher.publishEvent(entityTrailEvent);
        } catch (Exception e) {
            logger.error("tail publisher error", e);
        }
    }
}