package com.xforceplus.jpa.listener;


import com.xforceplus.entity.Role;
import com.xforceplus.event.dto.RoleCodeChanged;
import com.xforceplus.event.dto.RoleNameChanged;
import com.xforceplus.event.model.EntityPreSaveEvent;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Objects;


/**
 * 自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class RoleListener extends OperatorListener<Role> implements ApplicationEventPublisherAware {
    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    @PrePersist
    public void prePersist(Role entity) {
        //region 发送校验事件
        EntityPreSaveEvent<RoleCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, RoleCodeChanged.builder().entity(entity).build());
        applicationEventPublisher.publishEvent(codeValidationEvent);
        EntityPreSaveEvent<RoleNameChanged> nameValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, RoleNameChanged.builder().entity(entity).build());
        applicationEventPublisher.publishEvent(nameValidationEvent);
        //endregion
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (StringUtils.isBlank(entity.getRoleDesc())) {
            entity.setRoleDesc(StringUtils.EMPTY);
        }
        if (entity.getType() == null) {
            entity.setType(0);
        }
        if (entity.getFromRoleId() != null && entity.getFromRoleId() == 0) {
            entity.setFromRoleId(null);
        }
        super.beforeInsert(entity);
        this.cleanRelatedEntities(entity);
    }

    @PreUpdate
    public void preUpdate(Role entity) {
        if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            boolean codeChanged;
            boolean nameChanged;
            if (entityEntry != null) {
                codeChanged = !Objects.equals(entityEntry.getLoadedValue("code"), entity.getCode());
                nameChanged = !Objects.equals(entityEntry.getLoadedValue("name"), entity.getName());
            } else {
                codeChanged = entity.getCode() != null;
                nameChanged = entity.getName() != null;
            }
            if (codeChanged) {
                //region 发送校验事件
                EntityPreSaveEvent<RoleCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, RoleCodeChanged.builder().entity(entity).build());
                applicationEventPublisher.publishEvent(codeValidationEvent);
                //endregion
            }

            if (nameChanged) {
                //region 发送校验事件
                EntityPreSaveEvent<RoleNameChanged> nameValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, RoleNameChanged.builder().entity(entity).build());
                applicationEventPublisher.publishEvent(nameValidationEvent);
                //endregion
            }
        }
        if (StringUtils.isBlank(entity.getRoleDesc())) {
            entity.setRoleDesc(entity.getName());
        }
        if (entity.getFromRoleId() != null && entity.getFromRoleId() == 0) {
            entity.setFromRoleId(null);
        }
    }

    private void cleanRelatedEntities(Role entity) {
        entity.setRoleResourcesetRels(null);
        entity.setRoleUserRels(null);
        entity.setGradingOrgs(null);
        entity.setDefaultOrgs(null);
        entity.setFromRole(null);
    }
}