/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: StringQuery.java   2020-10-20 10-45-54
 * Author: Evan
 */
package com.xforceplus.data.query;

import org.springframework.data.domain.Sort;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.stream.Collectors;

import static com.xforceplus.api.utils.Separator.*;
import static com.xforceplus.data.query.QueryUtils.ORDER_BY_SQL;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 10-45-54
 * @since V1.0
 */
public class StringQuery {

    /**
     * 查询字符串
     */
    private StringBuilder stringQuery;
    /***
     * 查询参数
     */
    private NamedParams namedParams;

    /**
     * predicate
     * */
    private boolean predicate;


    /***
     * 默认构建函数
     */
    private StringQuery() {
        stringQuery = new StringBuilder();
        namedParams = NamedParams.instance();
        predicate = true;
    }

    /***
     * 构建对象
     * @return StringQuery
     */
    public static StringQuery builder() {
        return new StringQuery ();
    }

    /***
     * 查询参数
     * @param query  String
     * @return  StringQuery
     */
    public StringQuery query(String query) {
        if (this.predicate) {
            this.stringQuery.append(query);
        }
        return this;
    }

    /***
     * 参数设置
     * @param paramName
     * @param paramValue
     * @return StringQuery
     */
    public StringQuery param(String paramName, Object paramValue) {
        if (this.predicate) {
            namedParams.param(paramName, paramValue);
        }

        return this;
    }


    /***
     * 参数设置
     * @param paramName paramName
     * @param paramValue paramValue
     * @return StringQuery
     */
    public StringQuery paramDateStart(String paramName, Object paramValue) {
        if (this.predicate) {
            namedParams.paramDateStart(paramName, paramValue);
        }
        return this;
    }


    /***
     * 参数设置
     * @param paramName paramName
     * @param paramValue paramValue
     * @return StringQuery
     */
    public StringQuery paramDateEnd(String paramName, Object paramValue) {
        if (this.predicate) {
            namedParams.paramDateEnd(paramName, paramValue);
        }
        return this;
    }


    /**
     * like 自动 %%
     * @param paramName paramName
     * @param paramValue paramValue
     * @return StringQuery
     */
    public StringQuery likeParam(String paramName, Object paramValue) {
        if (this.predicate) {
            namedParams.likeParam(paramName, paramValue);
        }

        return this;
    }

    /**
     * like 'paramValue%'
     * @param paramName paramName
     * @param paramValue paramValue
     * @return StringQuery
     */
    public StringQuery likeEndParam(String paramName, Object paramValue) {
        if (this.predicate) {
            namedParams.likeEndParam(paramName, paramValue);
        }

        return this;
    }

    /***
     * like '%paramValue'
     * @param paramName paramName
     * @param paramValue paramValue
     * @return StringQuery
     */
    public StringQuery likeStartParam(String paramName, Object paramValue) {
        if (this.predicate) {
            namedParams.likeStartParam(paramName, paramValue);
        }
        return this;
    }

    /**
     * In
     * @param paramName paramName
     * @param paramValue paramValue
     * @return  StringQuery
     */
    public StringQuery inParam(String paramName, Object paramValue) {
        if (this.predicate) {
            namedParams.inParam(paramName, paramValue);
        }

        return this;
    }

    /**
     *  是否添加条件
     * @param predicate predicate
     * @return StringQuery
     */
    public StringQuery predicate(boolean predicate) {
        this.predicate = predicate;
        return this;
    }

    public StringQuery predicateNotNull(Object o) {
        if (o == null) {
            this.predicate = false;
        } else {
            this.predicate = true;
        }
        return this;
    }

    public StringQuery predicateIsNull(Object o) {
        if (o == null) {
            this.predicate = true;
        } else {
            this.predicate = false;
        }
        return this;
    }

    /***
     * 是否有文字
     * @param str 查询参数
     * @return StringQuery
     */
    public StringQuery predicateHasText(String str) {
        if (StringUtils.hasText(str)) {
            this.predicate = true;
        } else {
            this.predicate = false;
        }
        return this;
    }

    public StringQuery predicateNotHasText(String o) {
        if (StringUtils.hasText(o)) {
            this.predicate = false;
        } else {
            this.predicate = true;
        }
        return this;
    }

    public StringQuery predicateNotEmpty(Collection collection) {
        if (collection == null || collection.isEmpty()) {
            this.predicate = false;
        } else {
            this.predicate = true;
        }
        return this;
    }

    public StringQuery predicateIsEmpty(Collection collection) {
        if (collection == null || collection.isEmpty()) {
            this.predicate = true;
        } else {
            this.predicate = false;
        }
        return this;
    }

    public StringQuery predicateNotEmpty(Object[] array) {
        if (array == null || array.length == 0) {
            this.predicate = false;
        } else  {
            this.predicate = true;
        }
        return this;
    }

    public StringQuery predicateIsEmpty(Object[] array) {
        if (array == null || array.length == 0) {
            this.predicate = true;
        } else  {
            this.predicate = false;
        }
        return this;
    }

    /***
     * 排序方法
     * @param sort  排序
     * @return
     */
    public StringQuery sort(Sort sort) {
        if (!this.predicate) {
            return this;
        }
        if (sort == null || !sort.iterator().hasNext()) {
            return this;
        }
        this.stringQuery.append(sort.stream()
                .map(e -> e.getProperty() + SPACE + e.getDirection())
                .collect(Collectors.joining(COMMA, ORDER_BY_SQL, EMPTY)));
        return this;
    }

    /***
     * 自定义排序
     * @param sort 排序
     * @param sortQuery 自定义
     */
    public StringQuery sort(Sort sort, SortQuery sortQuery){
        if (!this.predicate) {
            return this;
        }
        // 排序
        if(sort==null||!sort.iterator ().hasNext ()){
            return this;
        }
        //查询排序的SQL
        String query=sortQuery.query (sort);
        if(!StringUtils.isEmpty (query)){
            stringQuery.append (query);
        }
        return this;
    }

    /***
     * 构建 getQuery
     * @return StringQuery
     */
    public StringQuery build() {
        this.predicate = true;
        return this;
    }

    /***
     * getQuery
     * @return SQL
     */
    public String getQuery() {
        return QueryUtils.removeWhereInvalidCondition(this.stringQuery);
    }

    /**
     * 获取参数
     * @return  NamedParams
     */
    public NamedParams getParams() {
        return this.namedParams;
    }
}
