package com.xforceplus.jpa.listener;


import com.xforceplus.entity.OrgStruct;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.data.jpa.essential.id.SnowflakeGenerator;
import io.geewit.utils.uuid.UUIDUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.util.CollectionUtils;

import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Calendar;
import java.util.List;
import java.util.Optional;


/**
 * 组织自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
@SuppressWarnings("all")
public class OrgListener extends OperatorListener<OrgStruct> implements ApplicationContextAware {
    private final static Logger logger = LoggerFactory.getLogger(OrgListener.class);

    private ApplicationContext applicationContext;

    @PrePersist
    public void prePersist(OrgStruct entity) {
        if (entity.getOrgId() == null) {
            long id = SnowflakeGenerator.id();
            entity.setOrgId(id);
        }
        if (entity.getParentId() == null || entity.getParentId() == 0) {
            entity.setOrgType(OrgType.GROUP);
        }
        if (entity.getCompanyId() != null && entity.getCompanyId() > 0) {
            entity.setOrgType(OrgType.COMPANY);
        }
        if (entity.getOrgType() == null) {
            entity.setOrgType(OrgType.NORMAL);
        }
        if (entity.getOrgCode() == null) {
            entity.setOrgCode(UUIDUtils.randomUUID());
        }
        JdbcTemplate jdbcTemplate = this.applicationContext.getBean(JdbcTemplate.class);
        Long codeCount = jdbcTemplate.queryForObject("select count(o.org_struct_id) from sys_org_struct o where o.tenant_id = ? and o.org_code = ?", Long.class, entity.getTenantId(), entity.getOrgCode());
        if(codeCount > 0) {
            throw new IllegalArgumentException("租户(" + entity.getTenantId() + ")下已存在重复的组织code(" + entity.getOrgCode() + ")");
        }
        if (entity.getOrgDesc() == null) {
            entity.setOrgDesc(entity.getOrgName());
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (entity.getAuditStatus() == null) {
            entity.setAuditStatus(1);
        }
        if (entity.getStatus() == 1) {
            entity.setEnabledTime(Calendar.getInstance().getTime());
        }
        this.preSave(entity);
        super.beforeInsert(entity);
    }

    @PreUpdate
    public void preUpdate(OrgStruct entity) {
        this.preSave(entity);
        if (entity.getCompanyId() != null && entity.getCompanyId() > 0) {
            entity.setOrgType(OrgType.COMPANY);
        }
        if (entity.getOrgType() == null) {
            entity.setOrgType(OrgType.NORMAL);
        }
        if (entity.getStatus() != null) {
            if (entity.getStatus() == 1) {
                entity.setEnabledTime(Calendar.getInstance().getTime());
            }
            if (entity.getStatus() == 0) {
                entity.setDisabledTime(Calendar.getInstance().getTime());
            }
        }
        super.beforeUpdate(entity);
    }

    private void preSave(OrgStruct entity) {
        if (entity.getCompanyId() == null) {
            entity.setCompanyId(0L);
        }
        if (entity.getParentId() == null) {
            //TODO
            entity.setParentId(0L);
        }
        this.buildParentIds(entity);
    }

    public void buildParentIds(OrgStruct entity) {
        if (!entity.builtParentIds) {
            StringBuilder builder = new StringBuilder();
            if (entity.getParentId() != null && entity.getParentId() > 0) {
                try {
                    JdbcTemplate jdbcTemplate = this.applicationContext.getBean(JdbcTemplate.class);
                    List<String> parentIdsList = jdbcTemplate.queryForList("select o.parent_ids from sys_org_struct o where o.org_struct_id = ?", String.class, entity.getParentId());
                    if (!CollectionUtils.isEmpty(parentIdsList)) {
                        Optional<String> parentIdsOptional = parentIdsList.stream().findAny();
                        if(parentIdsOptional.isPresent()) {
                            builder.append(parentIdsOptional.get());
                        } else {
                            String message = "不存在的组织parentId(" + entity.getParentId() + ")";
                            logger.warn(message);
                            throw new IllegalArgumentException(message);
                        }
                    }
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            }
            if (entity.getOrgId() != null) {
                builder.append(entity.getOrgId());
            }
            if (entity.getOrgType() != null) {
                builder.append(entity.getOrgType().separator());
            }
            entity.setParentIds(builder.toString());
            entity.builtParentIds = true;
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}