/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: QueryUtils.java   2020-10-20 11-07-03
 * Author: Evan
 */
package com.xforceplus.data.query;

import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.xforceplus.api.utils.Separator.EMPTY;
import static java.util.regex.Pattern.compile;

/**
 * <p>
 * Title: QueryUtils
 * </p>
 * <p>
 * Description: QueryUtils
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 11-07-03
 * @since V1.0
 */
public class QueryUtils {
    /***
     * 定义CountQuery {@value}
     */
    public static final String COUNT_QUERY_STRING = "select count(%s) from %s x";

    /**
     * ENTITY_NAME_NOT_BE_NULL {@value}
     */
    public static final String ENTITY_NAME_NOT_BE_NULL="Entity name must not be null or empty!";

    /**
     *QUERY_NOT_BE_NULL
     */
    public static final String QUERY_NOT_BE_NULL ="Query  must not be null!";

    /***
     *QUERY_HAS_TEXT
     */
    public static final String QUERY_HAS_TEXT="Query must has text!";
    /***
     * STRING_QUERY_HAS_TEXT
     */
    public static final String STRING_QUERY_HAS_TEXT="Query must has text!";

    /***
     *QUERY_COUNT_HAS_TEXT
     */
    public static final String QUERY_COUNT_HAS_TEXT="Query count must has text!";

    /***
     * QueryParams must not be null!
     */

    public static final String QUERY_PARAMS_NOT_BE_NULL = "QueryParams must not be null!";
    /**
     *
     */
    public static final String PAGEABLE_NOT_BE_NULL = "PageRequest must not be null!";
    /**
     *
     */
    public static final String TRANSFORMER_CLASS_MUST_NOT_BE_NULL = "Transformer Class must not be null";

    /**
     * 删除Order By
     */
    public static Pattern PATTERN_ORDER_BY = compile("order\\s*by[\\w|\\W|\\s|\\S]*", Pattern.CASE_INSENSITIVE);

    /***
     * 删除无效条件where  1=1 and
     */
    public static final String WHERE_INVALID_CONDITION_WITH_AND = "(where)\\s*(1\\s*=\\s*1)\\s*(and)";
    /**
     * 删除无效条件  where 1=1 and 正则表达式
     */
    public static final Pattern PATTERN_WHERE_INVALID_CONDITION_WITH_AND = compile(WHERE_INVALID_CONDITION_WITH_AND, Pattern.CASE_INSENSITIVE);

    /***
     * 删除无效条件where  1=1
     */
    public static final String WHERE_INVALID_CONDITION = "(where)\\s*(1\\s*=\\s*1)";
    /**
     * 删除无效条件  where 1=1 正则表达式
     */
    public static final Pattern PATTERN_WHERE_INVALID_CONDITION = compile(WHERE_INVALID_CONDITION, Pattern.CASE_INSENSITIVE);
    /**
     *  正则表达式：回车换行字符替换 ：{@value}
     */
    public static final String CRLF="(\\r\\n|\\r|\\n|\\n\\r)";
    /**
     * ORDER BY SQL :{@value}
     */
    public static final String ORDER_BY_SQL = " order by ";
    /**
     * fetch :{@value}
     */
    public static final String FETCH = " fetch";
    /**
     * FROM:{@value}
     */
    public static final String FROM = "from";
    /**
     * where :{@value}
     */
    public static final String WHERE = "where";


    /**
     * Returns the query string for the given class name.
     *
     * @param template   template
     * @param entityName entityName
     * @return String
     */
    public static String getQueryString(String template, String entityName) {

        Assert.hasText(entityName, ENTITY_NAME_NOT_BE_NULL);

        return String.format(template, entityName);
    }

    /**
     * 获取CountSQL
     *
     * @param entityName
     * @return
     */
    public static String getCountQueryString(String entityName) {
        String countQuery = String.format(COUNT_QUERY_STRING, "x", "%s");
        return getQueryString(countQuery, entityName);
    }

    /**
     * 查询SQL
     *
     * @param queryString queryString
     * @return String
     */
    public static String genCountQueryString(String queryString) {
        return "select count(*) " + removeSelect(queryString);
    }

    /**
     * <pre>
     * 去除JPQL语句前的select部分，用来生成查询总记录条数的HQL语句。
     *
     * <strong>程序范例：</strong>
     * String queryCountString = "select count(*) " + QueryUtils.removeSelect(queryString);
     * </pre>
     * @param queryString queryString
     * @return String
     */
    public static String removeSelect(String queryString) {
        Assert.hasText(queryString, QUERY_NOT_BE_NULL);
        queryString = removeFetch(queryString);
        queryString = removeOrders(queryString);
        int beginPos = queryString.toLowerCase().indexOf(FROM);
        Assert.isTrue(beginPos != -1, " the jpql : " + queryString + " must has a keyword 'from'");
        return queryString.substring(beginPos);
    }

    /**
     * 去掉orderBy语句,，用来生成查询总记录条数的HQL语句。
     * @param queryString queryString
     * @return String
     */
    public static String removeOrderBy(String queryString) {
        Assert.hasText(queryString, QUERY_NOT_BE_NULL);
        queryString = removeFetch(queryString);
        StringBuffer stringBuffer = new StringBuffer();
        Matcher matcher = PATTERN_ORDER_BY.matcher(queryString);
        while (matcher.find()) {
            matcher.appendReplacement(stringBuffer, EMPTY);
        }
        matcher.appendTail(stringBuffer);
        return stringBuffer.toString();
    }

    /**
     * <pre>
     * 去除HQL语句后的order by部分
     *
     * <strong>程序范例：</strong>
     * queryCountString = HqlUtils.removeOrders(queryCountString);
     *
     * </pre>
     *
     * @param queryString queryString
     * @return String
     */
    public static String removeOrders(String queryString) {
        Matcher matcher = PATTERN_ORDER_BY.matcher(queryString);
        StringBuffer stringBuffer = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(stringBuffer, EMPTY);
        }
        matcher.appendTail(stringBuffer);
        return stringBuffer.toString();
    }



    /**
     * 删除Where 后无效条件
     * @param query CharSequence
     * @return String  删除无效条件的结果
     */
    public static String removeWhereInvalidCondition(CharSequence query) {
        Assert.notNull(query,QUERY_NOT_BE_NULL);
        //替换换行回车
        String queryStr=String.valueOf(query).replaceAll(CRLF,EMPTY);
        //如果有where 除1=1 还有其它条件(where 1=1 and )的替换为 "where"
        Matcher matcher = PATTERN_WHERE_INVALID_CONDITION_WITH_AND.matcher(queryStr);
        StringBuffer buffer = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(buffer, WHERE);
        }
        matcher.appendTail(buffer);
        //删除无效条件 where 1=1 无其它条件的处理 替换空字符串
        matcher = PATTERN_WHERE_INVALID_CONDITION.matcher(buffer);
        StringBuffer bufferInvalid = new StringBuffer();
        while(matcher.find()){
            matcher.appendReplacement(bufferInvalid,EMPTY);
        }
        matcher.appendTail(bufferInvalid);
        return bufferInvalid.toString();
    }

    /**
     * <pre>
     * 去除JPQL语句内的fetch部分
     *
     * <strong>程序范例：</strong>
     * queryString = removeFetch(queryString);
     *
     * </pre>
     * @param queryString queryString
     * @return String
     */
    public static String removeFetch(String queryString) {
        Assert.hasText(queryString, QUERY_NOT_BE_NULL);
        return StringUtils.delete(queryString, FETCH);
    }
}
