package com.xforceplus.dao;

import com.xforceplus.entity.Company;
import com.xforceplus.entity.OrgStruct;
import io.geewit.data.jpa.essential.repository.JpaBatchRepository;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.*;
import org.springframework.data.repository.query.Param;
import org.springframework.lang.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;


/**
 * @author geewit
 * @since 2019-12-03
 */
public interface OrgStructDao extends JpaRepository<OrgStruct, Long>, JpaSpecificationExecutor<OrgStruct>, JpaBatchRepository<OrgStruct, Long>, CustomizedOrgStructDao {
    @Override
    @Modifying(clearAutomatically = true)
    @Query("delete from OrgStruct o where o.orgId = :orgId")
    void deleteById(@Param("orgId") Long id);

    @Override
    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.LOAD)
    Page<OrgStruct> findAll(@Nullable Specification<OrgStruct> spec, Pageable pageable);

    @Override
    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    List<OrgStruct> findAll(Specification<OrgStruct> specification);

    @Override
    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    List<OrgStruct> findAll(Specification<OrgStruct> specification, Sort sort);

    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    @Override
    Optional<OrgStruct> findById(Long orgId);

    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.parentIds like :parentIds% and o.status = 1")
    List<OrgStruct> findByParentIdsLike(@Param("parentIds") String parentIds);

    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.orgCode = :orgCode")
    List<OrgStruct> findByTenantIdAndOrgCode(@Param("tenantId") long tenantId, @Param("orgCode") String orgCode);

    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.orgCode in (:orgCodes)")
    List<OrgStruct> findByTenantIdAndOrgCodes(@Param("tenantId") long tenantId, @Param("orgCodes") Collection<String> orgCodes);

    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.orgId = :orgId and o.tenantId = :tenantId")
    OrgStruct findByTenantIdAndOrgId(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId and (o.parentId is null or o.parentId = 0)")
    List<OrgStruct> findRootsByTenantId(@Param("tenantId") long tenantId);

    @Query("select o.orgId from OrgStruct o where o.tenantId = :tenantId and o.status = 1 and (o.parentId is null or o.parentId = 0)")
    List<Long> findRootIdsByTenantId(@Param("tenantId") long tenantId);

    @Query("select o.company from OrgStruct o where o.tenantId = :tenantId")
    List<Company> findCompaniesByTenantId(@Param("tenantId") long tenantId);

    @Query("select count(distinct o.companyId) from OrgStruct o where o.tenantId =:tenantId and o.orgType = :#{T(com.xforceplus.tenant.security.core.domain.OrgType).COMPANY} and o.companyId > 0")
    int countCompaniesByTenantId(@Param("tenantId") long tenantId);

    @Query("select o.orgId from OrgStruct o where o.tenantId =:tenantId and o.orgCode = :orgCode")
    Long findIdByTenantIdAndCode(@Param("tenantId") long tenantId, @Param("orgCode") String orgCode);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 1, o.enabledTime = current_timestamp where o.orgId = :orgId")
    int enable(@Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 0, o.disabledTime = current_timestamp where o.orgId = :orgId")
    int disable(@Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 1, o.enabledTime = current_timestamp where o.orgId = :orgId and o.tenantId = :tenantId")
    int enable(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.status = 0, o.disabledTime = current_timestamp where o.orgId = :orgId and o.tenantId = :tenantId")
    int disable(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @Query("select o.orgId from OrgStruct o where o.tenantId =:tenantId and o.orgId in (:orgIds)")
    List<Long> findOrgIdsByTenantIdAndOrgIds(@Param("tenantId") long tenantId, @Param("orgIds") Collection<Long> orgIds);

    @Query(value = "select o.parent_ids from sys_org_struct o where o.org_struct_id =:orgId", nativeQuery = true)
    String findParentIdsByOrgId(@Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.companyId = :companyId, o.status = :status, o.auditStatus = :auditStatus, o.enabledTime = current_timestamp where o.orgId = :orgId")
    void approve(@Param("orgId") long orgId, @Param("companyId") long companyId, @Param("status") int status, @Param("auditStatus") int auditStatus);

    @Query("select o.orgName from OrgStruct o where o.orgId =:orgId")
    String findNameByOrgId(@Param("orgId") long orgId);

    @Modifying(flushAutomatically = true)
    @Query("update OrgStruct o set o.parentIds = :parentIds where o.orgId = :orgId")
    void updateParentIds(@Param("orgId") long orgId, @Param("parentIds") String parentIds);

    @EntityGraph(value = "Org.graph", type = EntityGraph.EntityGraphType.FETCH)
    @Query("select o from OrgStruct o where o.tenantId = :tenantId")
    List<OrgStruct> findByTenantId(@Param("tenantId") long tenantId);

    @Query("select count(o.orgId) from OrgStruct o where o.parentId = :orgId and o.status = 1")
    long countChildren(@Param("orgId") long orgId);

    @Query("select count(o.orgId) from OrgStruct o where o.tenantId = :tenantId and o.parentId = :orgId and o.status = 1")
    long countTenantChildren(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @Query("select o.orgId from OrgStruct o where o.tenantId = :tenantId and o.orgId = :orgId")
    Long findOrgIdByTenantIdAndOrgId(@Param("tenantId") long tenantId, @Param("orgId") long orgId);

    @Query("select count(o.orgId) from OrgStruct o where o.tenantId = :tenantId and o.status = 1 and o.parentId is not null")
    long countTenantWithoutRoot(@Param("tenantId") long tenantId);

    @Query("select o.orgId from OrgStruct o where o.tenantId = :tenantId and o.orgCode = :orgCode")
    List<Long> findOrgIdByTenantIdAndOrgCode(@Param("tenantId") long tenantId, @Param("orgCode") String orgCode);

    @Query(value = "select o.parent_ids from sys_org_struct o where o.org_struct_id in (:orgIds)", nativeQuery = true)
    Set<String> findParentIdsByOrgIds(@Param("orgIds") Collection<Long> orgIds);

    /**
     * 按组织Code
     * @param orgCode 查询列表
     * @return  Set<String>
     */
    @Query(value = "select o.parent_ids from sys_org_struct o where o.org_code=:orgCode", nativeQuery = true)
    Set<String> findParentIdsByOrgCode(@Param("orgCode") String orgCode);

    @Query("select o.orgId from OrgStruct o where o.parentIds like :parentIds and o.status = 1")
    Set<Long> findOrgIdsByParentIdsLike(@Param("parentIds") String parentIds);

    @Query("select o from OrgStruct o where o.tenantId = :tenantId and o.companyId = :companyId")
    List<OrgStruct> findByTenantIdAndCompanyId(@Param("tenantId") long tenantId, @Param("companyId") long companyId);

    @Query("select count(o) from OrgStruct o where o.parentId = :parentId and o.companyId = :companyId")
    long countCompaniesByParentIdAndCompanyId(@Param("parentId") long parentId, @Param("companyId") long companyId);

    @Query(value = "select o from OrgStruct o where o.orgId in (:orgIds)")
    List<OrgStruct> findByOrgIds(@Param("orgIds") Collection<Long> orgIds);

    List<OrgStruct> findByCompanyId(Long companyId);
}
