package com.xforceplus.dao.impl;

import com.xforceplus.api.model.OrgModel;
import com.xforceplus.dao.CustomizedOrgStructDao;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.query.OrgQueryHelper;
import io.geewit.data.jpa.essential.utils.TupleQueryUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.repository.query.QueryUtils;
import org.springframework.data.repository.support.PageableExecutionUtils;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Tuple;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import java.util.ArrayList;
import java.util.List;


@SuppressWarnings({"all"})
public class CustomizedOrgStructDaoImpl implements CustomizedOrgStructDao {

    @PersistenceContext
    private EntityManager entityManager;

    @Override
    public List<OrgStruct> findDescendantsById(long orgId) {
        TypedQuery subQuery = entityManager.createQuery("select o.parentIds from OrgStruct o where o.orgId = :orgId", String.class);
        subQuery.setParameter("orgId", orgId);
        subQuery.setFirstResult(0).setMaxResults(1);
        String parentIds = (String) subQuery.getSingleResult();
        if (StringUtils.isEmpty(parentIds)) {
            throw new IllegalArgumentException("未找到实体");
        }
        TypedQuery query = entityManager.createQuery("select o from OrgStruct o where o.parentIds like :parentIds order by o.parentIds", OrgStruct.class);
        query.setParameter("parentIds", parentIds + "%");
        return query.getResultList();
    }

    @Override
    public Page<Tuple> findTuples(OrgModel.Request.Query query, Pageable pageable) {
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Tuple> tupleCriteriaQuery = criteriaBuilder.createTupleQuery();
        Root<OrgStruct> tupleRoot = tupleCriteriaQuery.from(OrgStruct.class);
        Predicate predicate = OrgQueryHelper.queryTuplePredicate(query, tupleRoot, tupleCriteriaQuery, criteriaBuilder);
        if (predicate != null) {
            tupleCriteriaQuery.where(predicate);
        }
        Sort sort = pageable.isPaged() ? pageable.getSort() : Sort.unsorted();
        if (sort.isSorted()) {
            List<Order> orders = new ArrayList<>();
            if (!CollectionUtils.isEmpty(tupleCriteriaQuery.getOrderList())) {
                orders.addAll(tupleCriteriaQuery.getOrderList());
            }
            List<Order> toOrders = QueryUtils.toOrders(sort, tupleRoot, criteriaBuilder);
            if (!CollectionUtils.isEmpty(toOrders)) {
                orders.addAll(toOrders);
            }
            if (!CollectionUtils.isEmpty(orders)) {
                tupleCriteriaQuery.orderBy(orders);
            }
        }
        TypedQuery<Tuple> listQuery = entityManager.createQuery(tupleCriteriaQuery);
        if (pageable.isPaged()) {
            listQuery.setFirstResult((int) pageable.getOffset());
            listQuery.setMaxResults(pageable.getPageSize());
        }
        List<Tuple> results = listQuery.getResultList();
        CriteriaBuilder countCriteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Long> countCriteriaQuery = countCriteriaBuilder.createQuery(Long.class);
        Root<OrgStruct> countRoot = countCriteriaQuery.from(OrgStruct.class);
        Predicate countPredicate = OrgQueryHelper.queryCountPredicate(query, countRoot, countCriteriaQuery, countCriteriaBuilder);
        if (pageable.isPaged()) {
            return PageableExecutionUtils.getPage(results, pageable,
                    () -> TupleQueryUtils.executeCountQuery(TupleQueryUtils.getCountQuery(entityManager, countRoot, countCriteriaQuery, countCriteriaBuilder, countPredicate)));
        } else {
            return new PageImpl<>(results);
        }
    }
}
