/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: TenantUserExtendDaoImpl.java   2020-12-29 09-13-11
 * Author: Evan
 */
package com.xforceplus.dao.user.impl;

import com.xforceplus.bo.user.TenantUserBo;
import com.xforceplus.dao.user.TenantUserExtendDao;
import com.xforceplus.data.query.StringQuery;
import com.xforceplus.data.repository.AbstractDefaultJpaRepositoryImpl;
import com.xforceplus.dto.user.TenantAdminUserDTO;
import com.xforceplus.dto.user.TenantUserDTO;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Repository;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Set;

import static org.springframework.util.CollectionUtils.isEmpty;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-12-29 09-13-11
 * @since V1.0
 */
@Repository
public class TenantUserExtendDaoImpl extends AbstractDefaultJpaRepositoryImpl implements TenantUserExtendDao {
    /**
     * 租户管理员查询,限制返回数量200条
     */
    private static final String SQL_TENANT_ADMIN_USER = "SELECT srur.user_id,srur.tenant_id,srur.role_id FROM  " +
            " sys_role_user_rel srur ";

    /**
     * 列表信息查询数据
     */
    private static final String SQL_TENANT_USER = "SELECT su.user_id, su.tenant_id, su.`status` AS user_status, " +
            " su.user_type, su.user_name real_name, su.user_code, su.user_number, su.user_phone user_tel_phone, " +
            " su.create_time, su.account_id, ssa.username account_user_name, ssa.email account_email, " +
            " ssa.tel_phone AS account_tel_phone FROM sys_user su INNER JOIN sys_sass_account ssa on  su.account_id=ssa.account_id " +
            " WHERE   su.account_id = ssa.account_id  ";
    /**
     * 列表信息查询数据-子查询
     */
    private static final String SUB_SQL_TENANT_USER = " SELECT srur.user_id FROM sys_role_user_rel srur WHERE  1=1  ";


    /**
     * 根据租户ID及用户状态、用户类型查询相关信息
     *
     * @param pageable 分页对象
     * @param tenantUserBo 查询对象
     * @return Page<TenantUserDTO>
     */
    @Override
    public Page<TenantUserDTO> findTenantUsersByTenantId(Pageable pageable, TenantUserBo tenantUserBo) {
        StringQuery stringQuery = StringQuery.builder()
                .query(SQL_TENANT_USER)
                .predicate(true)
                .query(" AND su.tenant_id =:tenantId ")
                .param("tenantId", tenantUserBo.getTenantId())
                .predicateNotNull(tenantUserBo.getUserStatus())
                .query(" and su.status=:status").param("status", tenantUserBo.getUserStatus())
                .predicate(tenantUserBo.getUserType() != null)
                .query(" and su.user_id ")
                .predicate(tenantUserBo.getUserType() != null && tenantUserBo.getUserType().equals(1) & tenantUserBo.getTenantId() != null && tenantUserBo.getAdminRoleId() != null)
                .query(" not ")
                .predicate(tenantUserBo.getUserType() != null & tenantUserBo.getTenantId() != null && tenantUserBo.getAdminRoleId() != null)
                .query(" in ( ")
                .query(SUB_SQL_TENANT_USER)
                .predicate(tenantUserBo.getUserType() != null & tenantUserBo.getTenantId() != null && tenantUserBo.getAdminRoleId() != null)
                .query(" and  srur.tenant_id =:subTenantId  and srur.role_id=:subRoleId ")
                .param("subTenantId", tenantUserBo.getTenantId())
                .param("subRoleId", tenantUserBo.getAdminRoleId())
                .predicate(tenantUserBo.getUserType() != null)
                .query("  )")
                .build();
        // 使用下划线转换器
        return super.pagingSqlBy(pageable, stringQuery, TenantUserDTO.class, Boolean.TRUE);
    }

    /**
     * 根据租户ID及用户状态、用户类型查询相关信息
     *
     * @param pageable     分页对象
     * @param tenantUserBo 查询对象
     * @return Page<TenantUserDTO>
     */
    @Override
    public Page<TenantUserDTO> findTenantUsersByCondition(Pageable pageable, TenantUserBo tenantUserBo) {
        StringQuery stringQuery = StringQuery.builder()
                .query(SQL_TENANT_USER)
                .predicateNotEmpty(tenantUserBo.getTenantIds())
                .query(" AND su.tenant_id in(:tenantIds) ")
                .param("tenantIds", tenantUserBo.getTenantIds())
                .predicateNotNull(tenantUserBo.getTenantId())
                .query(" AND su.tenant_id=:tenantId")
                .param("tenantId", tenantUserBo.getTenantId())
                .predicateNotNull(tenantUserBo.getUserStatus())
                .query(" AND su.status=:status")
                .param("status", tenantUserBo.getUserStatus())
                .predicateNotNull(tenantUserBo.getCreatedDateBegin())
                .predicateNotNull(tenantUserBo.getCreatedDateEnd())
                .query(" AND su.create_time>=:beginDate and   su.create_time<:endDate")
                .param("beginDate", tenantUserBo.getCreatedDateBegin())
                .param("endDate", tenantUserBo.getCreatedDateEnd())
                .predicate(tenantUserBo.getUserType() != null)
                .query(" AND su.user_id ")
                .predicate(tenantUserBo.getUserType() != null && tenantUserBo.getUserType().equals(1) && tenantUserBo.getAdminRoleId() != null)
                .query(" not ")
                .predicate(tenantUserBo.getUserType() != null && tenantUserBo.getAdminRoleId() != null)
                .query(" in ( ")
                .query(SUB_SQL_TENANT_USER)
                .predicate(tenantUserBo.getUserType() != null && !isEmpty(tenantUserBo.getTenantIds()) && tenantUserBo.getAdminRoleId() != null)
                .query(" AND  srur.tenant_id in(:subTenantIds) and srur.role_id=:subRoleId ")
                .param("subTenantIds", tenantUserBo.getTenantIds())
                .param("subRoleId", tenantUserBo.getAdminRoleId())
                .predicate(tenantUserBo.getUserType() != null && isEmpty(tenantUserBo.getTenantIds()) && tenantUserBo.getAdminRoleId() != null)
                .query(" AND  srur.tenant_id =:subTenantId  and srur.role_id=:subRoleId ")
                .param("subTenantId", tenantUserBo.getTenantId())
                .param("subRoleId", tenantUserBo.getAdminRoleId())
                .predicate(tenantUserBo.getUserType() != null)
                .query("  )")
                .build();

        // 使用下划线转换器
        return super.pagingSqlBy(pageable, stringQuery, TenantUserDTO.class, Boolean.TRUE);
    }

    /**
     * 根据租户集合查询租户管理员数据
     *
     * @param tenantIds 租户Id集合
     * @param roleId    角色ID
     * @return List<TenantAdminUserDTO>
     */
    @Override
    public List<TenantAdminUserDTO> findTenantAdminUserByTenantIds(Set<Long> tenantIds, Long roleId) {
        Assert.notEmpty(tenantIds, "租户Ids不能为空");
        Assert.notNull(roleId, "角色Id不能为空");
        StringQuery stringQuery = StringQuery.builder()
                .query(SQL_TENANT_ADMIN_USER)
                .query(" WHERE srur.tenant_id in(:tenantIds) and role_id=:roleId limit 200 ")
                .predicate(true)
                .param("tenantIds", tenantIds)
                .param("roleId", roleId)
                .build();
        return this.findBySql(stringQuery, TenantAdminUserDTO.class, Boolean.TRUE);
    }

    /**
     * 根据根据租户ID和角色ID查询
     *
     * @param tenantId 租户ID
     * @param roleId   角色ID
     * @return List<TenantAdminUserDTO>
     */
    @Override
    public List<TenantAdminUserDTO> findTenantAdminUserByTenantId(Long tenantId, Long roleId) {
        Assert.notNull(tenantId, "租户Id不能为空");
        Assert.notNull(roleId,"角色Id不能为空");
        StringQuery stringQuery = StringQuery.builder()
                .query(SQL_TENANT_ADMIN_USER)
                .query(" WHERE srur.tenant_id=:tenantId and role_id=:roleId limit 200 ")
                .predicate(true)
                .param("tenantId", tenantId)
                .param("roleId", roleId)
                .build();
        return this.findBySql(stringQuery, TenantAdminUserDTO.class, Boolean.TRUE);
    }
}
