/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: AbstractJpaReositoryImpl.java   2020-10-20 10-41-23
 * Author: Evan
 */
package com.xforceplus.data.repository;

import com.xforceplus.data.query.NamedParams;
import com.xforceplus.data.query.QueryUtils;
import com.xforceplus.data.query.StringQuery;
import com.xforceplus.data.query.StringQueryUtils;
import com.xforceplus.data.transform.UnderlineToBeanResultTransformer;
import com.xforceplus.dto.user.TenantUserDTO;
import org.hibernate.query.internal.NativeQueryImpl;
import org.hibernate.transform.AliasToBeanResultTransformer;
import org.hibernate.transform.ResultTransformer;
import org.hibernate.transform.Transformers;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.Assert;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.util.List;

/**
 * <p>
 * Title: 扩展SpringDataJpa功能
 * </p>
 * <p>
 * Description:  扩展SpringDataJpa功能
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 10-41-23
 * @since V1.0
 */
public abstract class AbstractDefaultJpaRepositoryImpl {

    /**
     * 扩展类型
     */
    @PersistenceContext
    private EntityManager entityManager;

    /**
     * 查询
     * @param stringQuery StringQuery
     * @param clazz  Class<S>
     * @return List<S>
     */
    public <S> List<S> findBySql(final StringQuery stringQuery, final Class<S> clazz) {

        return findBySql( stringQuery, clazz, Boolean.FALSE);
    }


    public <S> List<S> findBySql(final StringQuery stringQuery, final Class<S> clazz, Boolean underline) {
        final Query query = this.entityManager.createNativeQuery(stringQuery.getQuery());
        AliasToBeanResultTransformer transformer;
        if (underline) {
            transformer = new UnderlineToBeanResultTransformer(clazz);
        } else {
            transformer = new AliasToBeanResultTransformer(clazz);
        }
        this.setQueryParams(query, stringQuery.getParams());
        query.unwrap(NativeQueryImpl.class).setResultTransformer(transformer);
        return query.getResultList();
    }
    /**
     * 分页查询
     * @param pageable 分页对象
     * @param sql SQL
     * @param params 参数
     * @param clazz 返回结果类
     * @return  Page<S>
     */
    public <S> Page<S> pagingSqlBy(final Pageable pageable, final String sql, final NamedParams params,
                                   final Class<S> clazz) {
        Assert.notNull(sql, QueryUtils.QUERY_HAS_TEXT);
        final String countSql = StringQueryUtils.genCountQueryString(sql);
        return this.pagingSqlBy(sql, countSql, params, pageable, clazz);
    }

    /**
     * 分页查询
     *
     * @param pageable  分页对象
     * @param stringQuery   StringQuery 查询对象
     * @param clazz     返回结果类
     * @param underline 自定义转换器，如果为null,则使用 AliasToBeanResultTransformer
     * @return Page<S>
     */
    public <S> Page<S> pagingSqlBy(final Pageable pageable,  StringQuery stringQuery,
                                   final Class<S> clazz, Boolean underline) {
        return this.pagingSqlBy(pageable, stringQuery.getQuery(), stringQuery.getParams(),clazz, underline);
    }

    /**
     * 分页查询
     *
     * @param pageable  分页对象
     * @param sql       SQL
     * @param params    参数
     * @param clazz     返回结果类
     * @param underline 自定义转换器，如果为null,则使用 AliasToBeanResultTransformer
     * @return Page<S>
     */
    public <S> Page<S> pagingSqlBy(final Pageable pageable, final String sql, final NamedParams params,
                                   final Class<S> clazz, Boolean underline) {
        Assert.notNull(sql, QueryUtils.QUERY_HAS_TEXT);
        final String countSql = StringQueryUtils.genCountQueryString(sql);
        return this.pagingSqlBy(sql, countSql, params, pageable, clazz, underline);
    }


    /**
     * 分页查询
     *
     * @param sql      分页查询SQL
     * @param countSql 计算总数
     * @param params   参数
     * @param pageable 分页对象
     * @param clazz    结类
     * @return Page<S>
     */
    @SuppressWarnings("unchecked")
    public <S> Page<S> pagingSqlBy(final String sql, final String countSql, final NamedParams params,
                                   final Pageable pageable, final Class<S> clazz, Boolean underline) {
        AliasToBeanResultTransformer transformer;
        if (underline) {
            transformer = new UnderlineToBeanResultTransformer(clazz);
        } else {
            transformer = new AliasToBeanResultTransformer(clazz);
        }
        Assert.hasText(sql, QueryUtils.QUERY_HAS_TEXT);
        Assert.hasText(countSql, QueryUtils.QUERY_COUNT_HAS_TEXT);
        Assert.notNull(params, QueryUtils.QUERY_PARAMS_NOT_BE_NULL);
        Assert.notNull(pageable, QueryUtils.PAGEABLE_NOT_BE_NULL);
        final Query query = this.entityManager.createNativeQuery(sql);
        this.setQueryParams(query, params);
        query.unwrap(NativeQueryImpl.class).setResultTransformer(transformer);
        query.setMaxResults(pageable.getPageSize());
        query.setFirstResult((int) pageable.getOffset());
        final List<S> resultList = query.getResultList();
        final Query countQuery =this.entityManager.createNativeQuery(countSql);
        this.setQueryParams(countQuery, params);
        final Long total = Long.valueOf(String.valueOf(countQuery.getSingleResult()));
        return new PageImpl<>(resultList, pageable, total);
    }

    /**
     * 分页查询
     * @param sql 分页查询SQL
     * @param countSql 计算总数
     * @param params 参数
     * @param pageable  分页对象
     * @param clazz 结类
     * @return Page<S>
     */
    @SuppressWarnings("unchecked")
    protected  <S> Page<S> pagingSqlBy(final String sql, final String countSql, final NamedParams params,
                                       final Pageable pageable, final Class<S> clazz) {
        //默认使用属性名转换器
        return this.pagingSqlBy(sql,countSql,params,pageable,clazz,Boolean.FALSE);
    }

    /***
     * 设置参数
     * @param query Query
     * @param params NamedParams
     */
    protected void setQueryParams(final Query query, final NamedParams params) {
        if (params.isEmpty()) {
            return;
        }
        params.getParameters().forEach((key, value) -> {
            query.setParameter(key, value);
        });
    }
}
