/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: UnderlineToBeanResultTransformer.java   2020-10-20 10-25-37
 * Author: Evan
 */
package com.xforceplus.data.transform;

import org.hibernate.HibernateException;
import org.hibernate.property.access.internal.PropertyAccessStrategyBasicImpl;
import org.hibernate.property.access.internal.PropertyAccessStrategyChainedImpl;
import org.hibernate.property.access.internal.PropertyAccessStrategyFieldImpl;
import org.hibernate.property.access.internal.PropertyAccessStrategyMapImpl;
import org.hibernate.property.access.spi.Setter;
import org.hibernate.transform.AliasToBeanResultTransformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.Arrays;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 10-25-37
 * @since V1.0
 */
@SuppressWarnings("all")
public class UnderlineToBeanResultTransformer extends AliasToBeanResultTransformer {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(UnderlineToBeanResultTransformer.class);
    /**
     * 错误提示消息
     */
    private static final String ERROR_RESULT_CLASS="Could not instantiate resultClass:";

    /**
     * 转换类
     */
    private final Class<?> resultClass;
    /**
     * 是否实始化
     */
    private boolean isInitialized;

    /**
     * aliases
     */
    private String[] aliases;
    /**
     * Setter
     */
    private Setter[] setters;

    /**
     * 默认构建函数
     *
     * @param resultClass 结果类
     */
    public <T> UnderlineToBeanResultTransformer(final Class<T> resultClass) {
        super(resultClass);
        Assert.notNull(resultClass, "resultClass cannot be null");
        this.isInitialized = false;
        this.resultClass = resultClass;
    }

    @Override
    public boolean isTransformedValueATupleElement(final String[] aliases, final int tupleLength) {
        return false;
    }

    @Override
    public Object transformTuple(final Object[] tuple, final String[] aliases) {
        Object result;
        try {
            if (!isInitialized) {
                initialize(aliases);
            } else {
                check(aliases);
            }
            result = resultClass.newInstance();
            for (int i = 0; i < aliases.length; i++) {
                if (setters[i] != null) {
                    Object value = tuple[i];
                    final Class<?>[] parameterTypes = setters[i].getMethod().getParameterTypes();
                    final Class<?> setterParameterType = parameterTypes[0];
                    if (value != null) {
                        if (setterParameterType == Long.class) {
                            value = Long.valueOf(value.toString());
                        } else if (setterParameterType == BigDecimal.class) {
                            value = new BigDecimal(value.toString());
                        } else if (setterParameterType == Boolean.class) {
                            value = Boolean.valueOf(value.toString());
                        }
                    }
                    setters[i].set(result, value, null);
                }
            }

        } catch (final InstantiationException | IllegalAccessException e) {
            throw new HibernateException(ERROR_RESULT_CLASS+resultClass.getName());
        }
        return result;
    }

    private void initialize(final String[] aliases) {
        final PropertyAccessStrategyChainedImpl propertyAccessStrategy = new PropertyAccessStrategyChainedImpl(
                PropertyAccessStrategyBasicImpl.INSTANCE, PropertyAccessStrategyFieldImpl.INSTANCE,
                PropertyAccessStrategyMapImpl.INSTANCE);
        this.aliases = new String[aliases.length];
        setters = new Setter[aliases.length];
        for (int i = 0; i < aliases.length; i++) {
            String alias = aliases[i];
            if (alias != null) {
                this.aliases[i] = alias;
                alias = UnderlineToBeanUtils.underlineToCamel(alias);
                try {
                    setters[i] = propertyAccessStrategy.buildPropertyAccess(resultClass, alias).getSetter();
                } catch (final Exception e) {
                    log.warn(resultClass + " property " + alias + " not found");
                }

            }
        }

        isInitialized = true;
    }

    private void check(final String[] aliases) {
        if (!Arrays.equals(aliases, this.aliases)) {
            String message="aliases are different from what is cached; aliases="+Arrays.asList(aliases)+"cached= "+Arrays.asList(this.aliases);
            throw new IllegalStateException(message);
        }
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final UnderlineToBeanResultTransformer that = (UnderlineToBeanResultTransformer) o;
        if (!resultClass.equals(that.resultClass)) {
            return false;
        }
        return Arrays.equals(aliases, that.aliases);
    }

    @Override
    public int hashCode() {
        int result = resultClass.hashCode();
        result = 31 * result + (aliases == null ? 0 : Arrays.hashCode(aliases));
        return result;
    }
}
