package com.xforceplus.query;

import com.xforceplus.api.model.UserModel.Request.Query;
import com.xforceplus.entity.*;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.util.CollectionUtils;

import javax.persistence.criteria.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author geewit
 */
@SuppressWarnings("all")
public class UserQueryHelper {
    private final static Logger logger = LoggerFactory.getLogger(UserQueryHelper.class);

    /**
     * 构建查询 Specification 对象
     *
     * @param query UserDto
     * @return Specification
     */
    public static Specification<User> querySpecification(Query query) {
        Specification<User> specification = (Specification<User>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            boolean joinTable = false;
            if (query.getExcludeBoundOrgVirtualNode() != null && query.getExcludeBoundOrgVirtualNode() && query.getTenantId() != null && query.getTenantId() > 0 && query.getOrgVirtualNodeId() != null && query.getOrgVirtualNodeId() > 0) {
                Subquery<OrgVirtualNodeUserRel> subquery = criteriaQuery.subquery(OrgVirtualNodeUserRel.class);
                Root relOrgVirtualTreeRoot = subquery.from(OrgVirtualNodeUserRel.class);
                subquery.select(relOrgVirtualTreeRoot);
                Predicate[] subPredicates = new Predicate[]{
                        builder.equal(relOrgVirtualTreeRoot.<Long>get("orgVirtualNodeId"), query.getOrgVirtualNodeId()),
                        builder.equal(root.<Long>get("id"), relOrgVirtualTreeRoot.<Long>get("userId"))
                };
                subquery.where(subPredicates);
                predicates.add(builder.not(builder.exists(subquery)));
            } else if (query.getExcludeBoundCurrent() != null && query.getTenantId() != null && query.getTenantId() > 0 && query.getExcludeBoundCurrent() && query.getRoleId() != null && query.getRoleId() > 0) {
                Subquery<RoleUserRel> subquery = criteriaQuery.subquery(RoleUserRel.class);
                Root roleUserRel = subquery.from(RoleUserRel.class);
                subquery.select(roleUserRel);
                Predicate[] subPredicates = new Predicate[]{
                        builder.equal(roleUserRel.<Long>get("roleId"), query.getRoleId()),
                        builder.equal(root.<Long>get("id"), roleUserRel.<Long>get("userId"))
                };
                subquery.where(subPredicates);
                predicates.add(builder.not(builder.exists(subquery)));
            } else if ((query.getRoleId() != null && query.getRoleId() > 0) || StringUtils.isNotBlank(query.getRoleCode()) || StringUtils.isNotBlank(query.getRoleIds())) {
                ListJoin<User, RoleUserRel> userJoinRoleUserRel = root.joinList("roleUserRels", JoinType.LEFT);
                if (query.getRoleId() != null && query.getRoleId() > 0) {
                    predicates.add(builder.equal(userJoinRoleUserRel.<Long>get("roleId"), query.getRoleId()));
                }
                if (StringUtils.isNotBlank(query.getRoleIds())) {
                    List<Long> roleIds = new ArrayList<>();
                    Arrays.stream(StringUtils.split(query.getRoleIds(), ",")).forEach(id -> {
                        try {
                            Long parseLong = Long.parseLong(id);
                            roleIds.add(parseLong);
                        } catch (NumberFormatException e) {
                            String message = e.getMessage() + ", id: " + id;
                            logger.warn(message);
                        }
                    });
                    if (!roleIds.isEmpty()) {
                        predicates.add(userJoinRoleUserRel.<Long>get("roleId").in(roleIds));
                    }
                }
                if (StringUtils.isNotBlank(query.getRoleCode())) {
                    Join<RoleUserRel, Role> joinRole = userJoinRoleUserRel.join("role", JoinType.LEFT);
                    predicates.add(builder.equal(joinRole.<String>get("code"), query.getRoleCode()));
                    if (query.getStatus() != null && query.getStatus() == 1) {
                        predicates.add(builder.equal(joinRole.get("status"), 1));
                    }
                }
                joinTable = true;
            } else if ((query.getOrgVirtualNodeId() != null && query.getOrgVirtualNodeId() > 0)) {
                ListJoin<User, OrgVirtualNodeUserRel> userJoinRoleOrgVirtualTreeRel = root.joinList("orgVirtualNodeUserRels", JoinType.LEFT);
                predicates.add(builder.equal(userJoinRoleOrgVirtualTreeRel.<Long>get("orgVirtualNodeId"), query.getOrgVirtualNodeId()));
            }
            if (StringUtils.isNotBlank(query.getAccountName())) {
                Join<User, Account> joinAccount = root.join("account", JoinType.LEFT);
                predicates.add(builder.or(
                        builder.equal(joinAccount.<String>get("username"), query.getAccountName()),
                        builder.equal(joinAccount.<String>get("telPhone"), query.getAccountName()),
                        builder.equal(joinAccount.<String>get("email"), query.getAccountName())
                ));
                if (query.getStatus() != null && query.getStatus() == 1) {
                    predicates.add(builder.equal(joinAccount.get("status"), 1));
                }
                joinTable = true;
            }

            if (query.getUserId() != null && query.getUserId() > 0) {
                predicates.add(builder.equal(root.<Long>get("id"), query.getUserId()));
            } else if (query.getUserIds() != null && !CollectionUtils.isEmpty(query.getUserIds())) {
                predicates.add(root.<Long>get("id").in(query.getUserIds()));
            }
            if (StringUtils.isNotBlank(query.getUserCode())) {
                predicates.add(builder.equal(root.<String>get("userCode"), query.getUserCode()));
            }
            if (StringUtils.isNotBlank(query.getUserName())) {
                predicates.add(builder.equal(root.<String>get("userName"), query.getUserName()));
            }
            if (StringUtils.isNotBlank(query.getUserNumber())) {
                predicates.add(builder.equal(root.<String>get("userNumber"), query.getUserNumber()));
            }
            if (query.getTenantId() != null && query.getTenantId() > 0) {
                predicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
            }
            if (StringUtils.isNotBlank(query.getUserPhone())) {
                predicates.add(builder.equal(root.<String>get("userPhone"), query.getUserPhone()));
            }
            if (StringUtils.isNotBlank(query.getUserEmailAddr())) {
                predicates.add(builder.equal(root.<String>get("userEmailAddr"), query.getUserEmailAddr()));
            }
            if (StringUtils.isNotBlank(query.getUserOptions())) {
                predicates.add(builder.or(
                        builder.like(root.get("userName"), query.getUserOptions() + "%"),
                        builder.equal(root.<String>get("userCode"), query.getUserOptions()),
                        builder.equal(root.<String>get("userNumber"), query.getUserOptions()),
                        builder.equal(root.<String>get("userPhone"), query.getUserOptions()),
                        builder.equal(root.<String>get("userEmailAddr"), query.getUserOptions())
                ));
            }
            if (StringUtils.isNotBlank(query.getTenantCode()) || StringUtils.isNotBlank(query.getTenantName())) {
                Join<User, Tenant> joinTenant = root.join("tenant", JoinType.LEFT);
                if (StringUtils.isNotBlank(query.getTenantCode())) {
                    predicates.add(builder.equal(joinTenant.<String>get("tenantCode"), query.getTenantCode()));
                }
                if (StringUtils.isNotBlank(query.getTenantName())) {
                    predicates.add(builder.like(joinTenant.get("tenantName"), query.getTenantName() + "%"));
                }
            }
            if (query.getStatus() != null) {
                predicates.add(builder.equal(root.<Integer>get("status"), query.getStatus()));
            }
            if (query.getAccountId() != null && query.getAccountId() > 0) {
                predicates.add(builder.equal(root.<Long>get("accountId"), query.getAccountId()));
            }
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            if (joinTable) {
                if (criteriaQuery.getResultType().isAssignableFrom(Long.class)) {
                    criteriaQuery.distinct(true);
                    return criteriaQuery.getRestriction();
                } else {
                    criteriaQuery.groupBy(root.<Long>get("id"));
                    return criteriaQuery.getGroupRestriction();
                }
            } else {
                return criteriaQuery.getRestriction();
            }
        };
        return specification;
    }

    /**
     * 构建查询Specification对象
     *
     * @param query UserDto
     * @return Specification
     */
    public static Specification<User> queryOneSpecification(Query query) {
        Specification<User> specification = (Specification<User>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (query.getUserId() != null && query.getUserId() > 0) {
                predicates.add(builder.equal(root.<Long>get("id"), query.getUserId()));
            }
            if (StringUtils.isNotBlank(query.getUserCode())) {
                predicates.add(builder.equal(root.<String>get("userCode"), query.getUserCode()));
            }
            if (query.getTenantId() != null && query.getTenantId() > 0) {
                predicates.add(builder.equal(root.<Long>get("tenantId"), query.getTenantId()));
            }
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return specification;
    }
}
