package com.xforceplus.utils;

import com.xforceplus.tenant.security.core.domain.OrgType;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 生成树
 *
 * @author geewit
 */
@SuppressWarnings("all")
public class OrgUtils {
    private final static Logger logger = LoggerFactory.getLogger(OrgUtils.class);

    /**
     * 根据parentIds 查询id集合
     * @param parentIds
     * @return
     */
    public static Set<Long> findOrgIdInParentIds(String parentIds) {
        if (StringUtils.isBlank(parentIds)) {
            return Collections.emptySet();
        }
        logger.debug("parentIds = " + parentIds);
        String pattern = "(\\d+)" + Arrays.stream(OrgType.values()).map(OrgType::separator).collect(Collectors.joining("|", "[", "]"));
        logger.debug("pattern = " + pattern);
        Matcher matcher = Pattern.compile(pattern).matcher(parentIds);
        Set<Long> parentOrgIds = new HashSet<>();
        while (matcher.find()) {
            String group = matcher.group(1);
            try {
                Long orgId = Long.parseLong(group);
                parentOrgIds.add(orgId);
            } catch (NumberFormatException e) {
                String message = e.getMessage() + ", value: " + group;
                logger.warn(message);
            }
        }
        if (logger.isInfoEnabled()) {
            logger.info("parentOrgIds = " + parentOrgIds.stream().map(String::valueOf).collect(Collectors.joining(",")));
        }
        return parentOrgIds;
    }

    public static Set<Long> findCompanyOrgIdsInParentIds(String parentIds) {
        logger.debug("parentIds = " + parentIds);
        String pattern = "(\\d+)" + OrgType.COMPANY.separator();
        logger.debug("pattern = " + pattern);
        Matcher matcher = Pattern.compile(pattern).matcher(parentIds);
        Set<Long> companyOrgIds = new HashSet<>();
        while (matcher.find()) {
            String group = matcher.group(1);
            try {
                long companyOrgId = Long.parseLong(group);
                companyOrgIds.add(companyOrgId);
                if (logger.isInfoEnabled()) {
                    logger.info("companyOrgId = " + companyOrgId);
                }
            } catch (NumberFormatException e) {
                String message = e.getMessage() + ", value: " + group;
                logger.warn(message);
            }
        }
        return companyOrgIds;
    }

    public static Long findRootIdInParentIds(String parentIds) {
        logger.debug("parentIds = " + parentIds);
        String pattern = "(\\d+)" + OrgType.GROUP.separator();
        logger.debug("pattern = " + pattern);
        Matcher matcher = Pattern.compile(pattern).matcher(parentIds);
        Long rootId = null;
        while (matcher.find()) {
            String group = matcher.group(1);
            try {
                rootId = Long.parseLong(group);
            } catch (NumberFormatException e) {
                String message = e.getMessage() + ", value: " + group;
                logger.warn(message);
            }
        }
        if (logger.isInfoEnabled()) {
            logger.info("rootId = " + rootId);
        }
        return rootId;
    }

    /**
     * 压缩parentIds集合减少sql次数
     * @param parentIdsCollection
     * @return
     */
    public static Set<String> compressParentIdsCollection(Collection<String> parentIdsCollection) {
        if (parentIdsCollection == null) {
            return Collections.emptySet();
        }
        if (parentIdsCollection.isEmpty()) {
            if (parentIdsCollection instanceof Set) {
                return (Set<String>)parentIdsCollection;
            } else {
                return Collections.emptySet();
            }
        }
        Stack<String> parentIdsStack = new Stack<>();
        //region 压缩parentIds集合减少sql次数
        parentIdsCollection.stream()
                .filter(StringUtils::isNotBlank)
                .sorted(String::compareTo).forEach(currentParentIds -> {
                    if (parentIdsStack.isEmpty()) {
                        parentIdsStack.push(currentParentIds);
                    } else {
                        String parentIds = parentIdsStack.pop();
                        if (!parentIds.equals(currentParentIds)) {
                            if (parentIds.startsWith(currentParentIds)) {
                                parentIdsStack.push(currentParentIds);
                            } else if (currentParentIds.startsWith(parentIds)) {
                                parentIdsStack.push(parentIds);
                            } else {
                                parentIdsStack.push(parentIds);
                                parentIdsStack.push(currentParentIds);
                            }
                        } else {
                            parentIdsStack.push(parentIds);
                        }
                    }
                });
        return parentIdsStack.stream().collect(Collectors.toSet());
    }

    /**
     * 过滤orgIds中不属于parentIdsCollection中的集合
     * @param orgIds
     * @param parentIdsCollection
     * @return
     */
    public static Set<Long> excludeInParentIds(Collection<Long> orgIds, Collection<String> parentIdsCollection) {
        Set<Long> filteredOrgIds;
        if (orgIds == null) {
            return Collections.emptySet();
        } else {
            if (orgIds instanceof Set) {
                filteredOrgIds = (Set<Long>)orgIds;
            } else {
                filteredOrgIds = new HashSet<>(orgIds);
            }
        }
        if (CollectionUtils.isEmpty(filteredOrgIds)) {
            return filteredOrgIds;
        }
        if (CollectionUtils.isEmpty(parentIdsCollection)) {
            return filteredOrgIds;
        }
        for (String parentIds : parentIdsCollection) {
            String filterParentIdsWithoutLastSperator = StringUtils.substring(parentIds, 0, parentIds.length() - 1);
            filteredOrgIds = filteredOrgIds.stream()
                    .filter(id -> !StringUtils.endsWith(filterParentIdsWithoutLastSperator, String.valueOf(id)))
                    .collect(Collectors.toSet());
        }
        return filteredOrgIds;
    }

    /**
     * 压缩 parentIds 集合 将 重复且包含的parentIds去掉
     * @param parentIdsCollection
     * @param orgIds
     * @return
     */
    public static Pair<Set<String>, Set<Long>> compressParentIdsCollectionAndFilterOrgIds(Collection<String> parentIdsCollection, Collection<Long> orgIds) {
        Set<Long> filteredOrgIds;
        if (orgIds == null) {
            filteredOrgIds = Collections.emptySet();
        } else {
            if (orgIds instanceof Set) {
                filteredOrgIds = (Set<Long>)orgIds;
            } else {
                filteredOrgIds = new HashSet<>(orgIds);
            }
        }

        Stack<String> parentIdsStack = new Stack<>();
        final Set<Long> finalFilteredOrgIds = filteredOrgIds;
        //region 压缩parentIds集合减少sql次数
        parentIdsCollection.stream()
                .filter(StringUtils::isNotBlank)
                .sorted(String::compareTo).forEach(currentParentIds -> {
                    if (parentIdsStack.isEmpty()) {
                        parentIdsStack.push(currentParentIds);
                    } else {
                        String parentIds = parentIdsStack.pop();
                        OrgUtils.filterOrgIdsInParentIds(finalFilteredOrgIds, currentParentIds);
                        if (!parentIds.equals(currentParentIds)) {
                            if (parentIds.startsWith(currentParentIds)) {
                                parentIdsStack.push(currentParentIds);
                            } else if (currentParentIds.startsWith(parentIds)) {
                                parentIdsStack.push(parentIds);
                            } else {
                                parentIdsStack.push(parentIds);
                                parentIdsStack.push(currentParentIds);
                            }
                        } else {
                            parentIdsStack.push(parentIds);
                        }
                    }
                });
        Set<String> parentIdsSet = parentIdsStack.stream().collect(Collectors.toSet());

        return Pair.of(parentIdsSet, filteredOrgIds);
    }

    /**
     * 过滤在 like 条件中和 equal 重复的 includeOrgIds, includeOrgIds只保留于like条件不重合的条目
     * @param orgIds
     * @param parentIds
     */
    public static void filterOrgIdsInParentIds(Set<Long> orgIds, String parentIds) {
        if (CollectionUtils.isNotEmpty(orgIds) && StringUtils.isNotBlank(parentIds)) {
            String filterParentIdsWithoutLastSperator = StringUtils.substring(parentIds, 0, parentIds.length() - 1);
            Set<Long> filteredOrgIds = orgIds.stream()
                    .filter(id -> !StringUtils.endsWith(filterParentIdsWithoutLastSperator, String.valueOf(id)))
                    .collect(Collectors.toSet());
            orgIds.clear();
            orgIds.addAll(filteredOrgIds);
        }
    }
}
