package com.xforceplus.utils;

import com.google.common.base.Splitter;
import com.xforceplus.tenant.security.core.domain.OrgType;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 生成树
 *
 * @author geewit
 */
@SuppressWarnings("all")
public class OrgUtils {
    private final static Logger logger = LoggerFactory.getLogger(OrgUtils.class);

    private static final Pattern SPLIT_PATTERN_OF_ORG_PARENT_IDS = Pattern.compile(Arrays.stream(OrgType.values())
            .map(OrgType::separator)
            .collect(Collectors.joining("|", "[", "]")));

    private static final Map<OrgType, Pattern> SPLIT_PATTERN_OF_ORG_PARENT_IDS_MAP = Arrays.stream(OrgType.values())
            .collect(Collectors.toMap(Function.identity(), orgType -> Pattern.compile("(\\d+)[" + orgType.separator() + "]")));

    /**
     * 根据parentIds 查询id集合
     * @param parentIds
     * @return
     */
    public static Set<Long> findOrgIdInParentIds(String parentIds) {
        if (StringUtils.isBlank(parentIds)) {
            return Collections.emptySet();
        }
        Set<Long> orgIds = Splitter.on(SPLIT_PATTERN_OF_ORG_PARENT_IDS).omitEmptyStrings().trimResults()
                .splitToStream(parentIds).filter(NumberUtils::isDigits)
                .map(v -> {
                    try {
                        return Long.parseLong(v);
                    } catch (NumberFormatException e) {
                        logger.info("{}, value: {}", e.getMessage(), v);
                        return null;
                    }
                }).filter(Objects::nonNull)
                .collect(Collectors.toSet());
        return orgIds;
    }

    public static Set<Long> findOrgIdsInParentIdsByOrgType(String parentIds, OrgType orgType, boolean findFirst) {
        logger.debug("parentIds = " + parentIds);
        if (StringUtils.isBlank(parentIds)) {
            return Collections.emptySet();
        }
        Matcher matcher = SPLIT_PATTERN_OF_ORG_PARENT_IDS_MAP.get(orgType).matcher(parentIds);
        Set<Long> orgIds = new HashSet<>();
        while (matcher.find()) {
            String group = matcher.group(1);
            try {
                long orgId = Long.parseLong(group);
                orgIds.add(orgId);
                if (logger.isDebugEnabled()) {
                    logger.debug("orgId = " + orgId);
                }
                if (findFirst && orgId > 0) {
                    break;
                }
            } catch (NumberFormatException e) {
                logger.info("{}, value: {}", e.getMessage(), group);
            }
        }
        return orgIds;
    }

    public static Set<Long> findCompanyOrgIdsInParentIds(String parentIds) {
        Set<Long> orgIds = OrgUtils.findOrgIdsInParentIdsByOrgType(parentIds, OrgType.COMPANY, false);
        return orgIds;
    }

    public static Long findRootIdInParentIds(String parentIds) {
        Set<Long> orgIds = OrgUtils.findOrgIdsInParentIdsByOrgType(parentIds, OrgType.GROUP, true);
        return orgIds.stream().findFirst().orElseGet(null);
    }

    /**
     * 压缩parentIds集合减少sql次数
     * @param parentIdsCollection
     * @return
     */
    public static Set<String> compressParentIdsCollection(Collection<String> parentIdsCollection) {
        if (parentIdsCollection == null) {
            return Collections.emptySet();
        }
        if (parentIdsCollection.isEmpty()) {
            if (parentIdsCollection instanceof Set) {
                return (Set<String>)parentIdsCollection;
            } else {
                return Collections.emptySet();
            }
        }
        Stack<String> parentIdsStack = new Stack<>();
        //region 压缩parentIds集合减少sql次数
        parentIdsCollection.stream()
                .filter(StringUtils::isNotBlank)
                .sorted(String::compareTo).forEach(currentParentIds -> {
                    if (parentIdsStack.isEmpty()) {
                        parentIdsStack.push(currentParentIds);
                    } else {
                        String parentIds = parentIdsStack.pop();
                        if (!parentIds.equals(currentParentIds)) {
                            if (parentIds.startsWith(currentParentIds)) {
                                parentIdsStack.push(currentParentIds);
                            } else if (currentParentIds.startsWith(parentIds)) {
                                parentIdsStack.push(parentIds);
                            } else {
                                parentIdsStack.push(parentIds);
                                parentIdsStack.push(currentParentIds);
                            }
                        } else {
                            parentIdsStack.push(parentIds);
                        }
                    }
                });
        return parentIdsStack.stream().collect(Collectors.toSet());
    }

    /**
     * 过滤orgIds中不属于parentIdsCollection中的集合
     * @param orgIds
     * @param parentIdsCollection
     * @return
     */
    public static Set<Long> excludeInParentIds(Collection<Long> orgIds, Collection<String> parentIdsCollection) {
        Set<Long> filteredOrgIds;
        if (orgIds == null) {
            return Collections.emptySet();
        } else {
            if (orgIds instanceof Set) {
                filteredOrgIds = (Set<Long>)orgIds;
            } else {
                filteredOrgIds = new HashSet<>(orgIds);
            }
        }
        if (CollectionUtils.isEmpty(filteredOrgIds)) {
            return filteredOrgIds;
        }
        if (CollectionUtils.isEmpty(parentIdsCollection)) {
            return filteredOrgIds;
        }
        for (String parentIds : parentIdsCollection) {
            String filterParentIdsWithoutLastSperator = StringUtils.substring(parentIds, 0, parentIds.length() - 1);
            filteredOrgIds = filteredOrgIds.stream()
                    .filter(id -> !StringUtils.endsWith(filterParentIdsWithoutLastSperator, String.valueOf(id)))
                    .collect(Collectors.toSet());
        }
        return filteredOrgIds;
    }

    /**
     * 压缩 parentIds 集合 将 重复且包含的parentIds去掉
     * @param parentIdsCollection
     * @param orgIds
     * @return
     */
    public static Pair<Set<String>, Set<Long>> compressParentIdsCollectionAndFilterOrgIds(Collection<String> parentIdsCollection, Collection<Long> orgIds) {
        Set<Long> filteredOrgIds;
        if (orgIds == null) {
            filteredOrgIds = Collections.emptySet();
        } else {
            if (orgIds instanceof Set) {
                filteredOrgIds = (Set<Long>)orgIds;
            } else {
                filteredOrgIds = new HashSet<>(orgIds);
            }
        }

        Stack<String> parentIdsStack = new Stack<>();
        final Set<Long> finalFilteredOrgIds = filteredOrgIds;
        //region 压缩parentIds集合减少sql次数
        parentIdsCollection.stream()
                .filter(StringUtils::isNotBlank)
                .sorted(String::compareTo).forEach(currentParentIds -> {
                    if (parentIdsStack.isEmpty()) {
                        parentIdsStack.push(currentParentIds);
                    } else {
                        String parentIds = parentIdsStack.pop();
                        OrgUtils.filterOrgIdsInParentIds(finalFilteredOrgIds, currentParentIds);
                        if (!parentIds.equals(currentParentIds)) {
                            if (parentIds.startsWith(currentParentIds)) {
                                parentIdsStack.push(currentParentIds);
                            } else if (currentParentIds.startsWith(parentIds)) {
                                parentIdsStack.push(parentIds);
                            } else {
                                parentIdsStack.push(parentIds);
                                parentIdsStack.push(currentParentIds);
                            }
                        } else {
                            parentIdsStack.push(parentIds);
                        }
                    }
                });
        Set<String> parentIdsSet = parentIdsStack.stream().collect(Collectors.toSet());

        return Pair.of(parentIdsSet, filteredOrgIds);
    }

    /**
     * 过滤在 like 条件中和 equal 重复的 includeOrgIds, includeOrgIds只保留于like条件不重合的条目
     * @param orgIds
     * @param parentIds
     */
    public static void filterOrgIdsInParentIds(Set<Long> orgIds, String parentIds) {
        if (CollectionUtils.isNotEmpty(orgIds) && StringUtils.isNotBlank(parentIds)) {
            String filterParentIdsWithoutLastSperator = StringUtils.substring(parentIds, 0, parentIds.length() - 1);
            Set<Long> filteredOrgIds = orgIds.stream()
                    .filter(id -> !StringUtils.endsWith(filterParentIdsWithoutLastSperator, String.valueOf(id)))
                    .collect(Collectors.toSet());
            orgIds.clear();
            orgIds.addAll(filteredOrgIds);
        }
    }
}
