package com.xforceplus.domain.cloudshell;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonRawValue;
import io.swagger.v3.oas.annotations.media.Schema;
import java.util.Objects;
import java.util.StringJoiner;

/**
 * 用以sse消息的通知,有两种可能.<br>
 * progressChange 表示 value记录的是最新的进度浮点数.
 * finished 表示进度已经结束,这是最终结果.
 *
 * @author dongbin
 * @version 0.1 2023/6/8 17:51
 * @since 1.8
 */
@Schema(name = "sse消息通知")
public class SseNotification {

    /**
     * 进度发生了改变.value记录了最新的进度.
     */
    public static final String PROGRESS_CHANGE_TYPE = "progressChange";

    /**
     * 进度已经走完, value记录了结果.
     */
    public static final String FINISHED_TYPE = "progressFinished";

    /**
     * 进度无法完成,发生了错误.
     */
    public static final String ERROR_FINISHED_TYPE = "progressError";

    @Schema(description = "通知类型.", allowableValues = {PROGRESS_CHANGE_TYPE, FINISHED_TYPE}, format = "string")
    private final String type;

    @Schema(description = "进度值, 当type=progressChange时才有效.", minimum = "0", maximum = "1.0", format = "float")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final Float progress;

    @Schema(
        description = "进度结束后的值, 当type=progressFinished时才有效.",
        format = "json"
    )
    @JsonRawValue
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final String info;

    @Schema(description = "错误信息, 当type=progressError时才有效.")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final String cause;

    /**
     * 创建一个进度改变的通知.
     */
    public static SseNotification buildProgressChange(float progress) {
        return new SseNotification(progress);
    }

    /**
     * 创建一个进度完成通知.
     */
    public static SseNotification buildFinished(String value) {
        return new SseNotification(value);
    }

    /**
     * 创建一个表示进度条错误中止.
     *
     * @param throwable 原因.
     * @return 实例.
     */
    public static SseNotification buildError(Throwable throwable) {
        return new SseNotification(throwable);
    }

    private SseNotification(float progress) {
        this.type = PROGRESS_CHANGE_TYPE;
        this.progress = progress;
        // null表示不会出现在最终的输出中.
        this.info = null;
        this.cause = null;
    }

    private SseNotification(String info) {
        this.type = FINISHED_TYPE;
        // null表示不会出现在最终的输出中.
        this.progress = null;
        if (info == null) {
            this.info = "{}";
        } else {
            this.info = info;
        }
        this.cause = null;
    }

    private SseNotification(Throwable throwable) {
        this.type = ERROR_FINISHED_TYPE;
        this.progress = null;
        this.info = null;
        if (throwable == null) {
            this.cause = "Unknown Error!";
        } else {
            this.cause = throwable.getMessage();
        }
    }

    public String getType() {
        return type;
    }

    public Float getProgress() {
        return progress;
    }

    public Object getInfo() {
        return info;
    }

    public String getCause() {
        return cause;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof SseNotification)) {
            return false;
        }
        SseNotification that = (SseNotification) o;
        return Objects.equals(getType(), that.getType()) &&
            Objects.equals(getProgress(), that.getProgress()) &&
            Objects.equals(getInfo(), that.getInfo()) && Objects.equals(cause, that.cause);
    }

    @Override
    public int hashCode() {
        return Objects.hash(getType(), getProgress(), getInfo(), cause);
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", SseNotification.class.getSimpleName() + "[", "]")
            .add("cause='" + cause + "'")
            .add("info='" + info + "'")
            .add("progress=" + progress)
            .add("type='" + type + "'")
            .toString();
    }

}
