package com.xforceplus.domain.org;

import com.fasterxml.jackson.annotation.*;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.xforceplus.domain.company.Switch;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.tenant.security.core.domain.IOperator;
import com.xforceplus.tenant.security.core.domain.Org;
import com.xforceplus.tenant.security.token.domain.view.RedisView;
import io.geewit.core.jackson.view.View;
import io.geewit.core.utils.enums.EnumMapUtils;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.FieldNameConstants;
import org.springframework.format.annotation.DateTimeFormat;

import java.util.*;


/**
 * 组织
 *
 * @author geewit
 */
@Schema(name = "组织Dto")
@FieldNameConstants
@Setter
@Getter
public class OrgDto<O extends OrgDto<O>> extends Org<O> implements IOperator {
    /**
     * 组织描述
     */
    @JsonView({View.class, RedisView.class})
    protected String orgDesc;

    /**
     * 组织业务类型
     */
    @JsonView({View.class, RedisView.class})
    protected String orgBizType;

    /**
     * 租户名称
     */
    @JsonView({View.List.class, RedisView.class})
    protected String tenantName;

    /**
     * 租户code
     */
    @JsonView({View.List.class, RedisView.class})
    protected String tenantCode;

    /**
     * 上级组织名称
     */
    @JsonView({View.class, RedisView.class})
    protected String parentName;

    /**
     * 上级组织code
     */
    @JsonView({View.class, RedisView.class})
    protected String parentCode;

    /**
     * 关联租户开关组合
     */
    @JsonView({View.class, RedisView.class})
    protected Integer relatedSwitches;

    /**
     * 启用状态 1:启用, 0:注销
     */
    @JsonView({View.class, RedisView.class})
    protected Integer status;

    /**
     * 公司审核状态 0:待审核，1:审核通过, 2:审核失败
     */
    @JsonView({View.class, RedisView.class})
    protected Integer auditStatus;

    /**
     * 状态启用时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date enabledTime;

    /**
     * 状态注销时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date disabledTime;

    /**
     * 创建时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date createTime;

    /**
     * 创建人id
     */
    @JsonView({View.class, RedisView.class})
    protected String createrId;

    /**
     * 创建人名称
     */
    @JsonView({View.class, RedisView.class})
    protected String createrName;

    /**
     * 更新人id
     */
    @JsonView({View.class, RedisView.class})
    protected String updaterId;

    /**
     * 更新人名称
     */
    @JsonView({View.class, RedisView.class})
    protected String updaterName;

    /**
     * 更新时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date updateTime;

    /**
     * 归属用户数量
     */
    @JsonView({View.class})
    protected Long userCount;

    /**
     * 子节点数量
     */
    @JsonView({View.class})
    protected Long childrenCount;

    /**
     * 用户绑定勾选状态
     */
    @JsonView({View.class, RedisView.class})
    protected Boolean userBound;

    /**
     * 公司代码集合
     */
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @JsonView({View.class, RedisView.class})
    protected Set<String> companyNos;

    /**
     * 组织扩展信息
     */
    @ArraySchema(schema = @Schema(implementation = OrgExtensionDto.class))
    @JsonView({View.class, RedisView.class})
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    protected Set<OrgExtensionDto> extensions;

    /**
     * 父级组织集合
     */
    @ArraySchema(schema = @Schema(implementation = OrgDto.class))
    @JsonView({View.class, RedisView.class})
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    protected List<O> parentOrgs;

    /**
     * 后辈集合
     */
    @ArraySchema(schema = @Schema(implementation = OrgDto.class))
    @JsonView(View.class)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    protected List<O> descendants;

    /**
     * 祖辈集合
     */
    @ArraySchema(schema = @Schema(implementation = OrgDto.class))
    @JsonView(View.class)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    protected List<O> ancestors;

    /**
     * 组织类型code
     */
    @JsonView({View.List.class, View.Tree.class, RedisView.class})
    protected String orgTypeCode;

    /**
     * 组织类型名称
     */
    @JsonView({View.List.class, View.Tree.class, RedisView.class})
    protected String orgTypeName;

    /**
     * 指定用户是否是当前组织节点管理员
     */
    @JsonView({View.Tree.class, View.List.class})
    protected Boolean isOrgAdmin;

    /**
     * 是否选中
     */
    @JsonView({View.Tree.class, View.List.class})
    protected Boolean checked;

    @ArraySchema(schema = @Schema(implementation = UserDto.class))
    @JsonView(View.List.class)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    protected List<UserDto> orgUsers;

    /**
     * 是否忽略实体重置parentIds监听器
     */
    @JsonIgnore
    protected Boolean ignoreRebuildParentIdsListener;

    /**
     * 是否忽略实体变更校验监听器
     */
    @JsonIgnore
    protected Boolean ignoreChangeValidListener;

    /**
     * 是否忽略实体变更更新缓存监听器
     */
    @JsonIgnore
    protected Boolean ignoreChangeCacheListener;

    @JsonView({View.class, RedisView.class})
    @JsonSerialize(using = ToStringSerializer.class)
    @Override
    public Long getOrgId() {
        return super.id;
    }

    @JsonView({View.class, RedisView.class})
    @JsonSerialize(using = ToStringSerializer.class)
    @Override
    public Long getTenantId() {
        return tenantId;
    }

    @JsonView({View.class, RedisView.class})
    @JsonSerialize(using = ToStringSerializer.class)
    @Override
    public Long getCompanyId() {
        return companyId;
    }

    @JsonView({View.class, RedisView.class})
    @JsonSerialize(using = ToStringSerializer.class)
    @Override
    public Long getParentId() {
        return parentId;
    }

    @JsonView({View.class, RedisView.class})
    @Override
    public String getTaxNum() {
        return super.taxNum;
    }

    @JsonView({View.class, RedisView.class})
    @Override
    public String getCompanyName() {
        return super.companyName;
    }

    /**
     * @return 公司code
     */
    @Override
    @JsonView({View.List.class, RedisView.class})
    public String getCompanyCode() {
        return super.companyCode;
    }

    /**
     * 用户对应的组织关系类型
     */
    @JsonView({View.Tree.class, View.List.class})
    public Integer getUserOrgType() {
        return super.sign;
    }

    /**
     * @return 枚举化的关联租户开关组合Map
     */
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @JsonView(View.List.class)
    public Map<Switch, Boolean> getRelatedSwitchesMap() {
        if (this.relatedSwitches == null) {
            return Collections.emptyMap();
        }
        return EnumMapUtils.toEnumMap(Switch.class, this.relatedSwitches);
    }

    public void setId(Long id) {
        super.id = id;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof OrgDto)) {
            return false;
        }
        OrgDto<?> that = (OrgDto<?>) o;
        return Objects.equals(super.id, that.id) || (Objects.equals(super.tenantId, that.tenantId) && Objects.equals(super.orgCode, that.orgCode));
    }

    @Override
    public int hashCode() {
        if (super.id != null && super.id > 0) {
            return Objects.hash(super.id);
        }
        return Objects.hash(super.tenantId, super.orgCode);
    }

    @Override
    public String toString() {
        return "OrgDto{" + "orgCode=" + orgCode +
                ", orgName=" + orgName +
                ", tenantId=" + tenantId +
                ", companyId=" + companyId +
                ", orgType=" + orgType +
                ", taxNum=" + taxNum +
                ", id=" + id +
                ", parentId=" + parentId +
                ", parentIds=" + parentIds +
                '}';
    }
}
