package com.xforceplus.domain.resource;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.domain.resource.view.ServicePackageView;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.tenant.security.core.domain.IOperator;
import com.xforceplus.tenant.security.token.domain.view.RedisView;
import io.geewit.core.jackson.view.View;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Schema;
import java.util.Date;
import java.util.Objects;
import java.util.Set;
import javax.validation.constraints.Max;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.FieldNameConstants;
import org.hibernate.validator.constraints.Range;
import org.springframework.format.annotation.DateTimeFormat;

/**
 * 服务包
 * @author geewit
 */
@Schema(name = "服务包Dto")
@FieldNameConstants
@Setter
@Getter
public class ServicePackageDto implements IOperator {
    public ServicePackageDto() {
    }

    public ServicePackageDto(Long servicePackageId, String servicePackageName, Long appId) {
        this.servicePackageId = servicePackageId;
        this.servicePackageName = servicePackageName;
        this.appId = appId;
    }

    public ServicePackageDto(Long servicePackageId, String servicePackageCode, String servicePackageName, Long appId,
                             Integer type) {
        this.servicePackageId = servicePackageId;
        this.servicePackageCode = servicePackageCode;
        this.servicePackageName = servicePackageName;
        this.appId = appId;
        this.type = type;
    }

    @ArraySchema(schema = @Schema(implementation = ResourcesetDto.class))
    @JsonView(ServicePackageView.ServicePackageInfo.class)
    protected Set<ResourcesetDto> resourcesets;

    /**
     * 服务包id
     */
    @JsonView({View.class, RedisView.class})
    protected Long servicePackageId;

    /**
     * 应用id
     */
    @JsonView({View.class, RedisView.class})
    @Min(1)
    @NotNull(groups = ValidationGroup.OnCreate.class)
    protected Long appId;

    /**
     * 应用名称
     */
    @JsonView({View.class})
    protected String appName;

    /**
     * 服务包code
     */
    @JsonView({View.class, RedisView.class})
    @Pattern(regexp = "\\s*|[a-zA-Z\\d_\\-]+", groups = {ValidationGroup.OnCreate.class, ValidationGroup.OnUpdate.class}, message = "代码不能有汉字")
    protected String servicePackageCode;

    /**
     * 服务包名称
     */
    @JsonView({View.class, RedisView.class})
    @NotEmpty(groups = ValidationGroup.OnCreate.class)
    protected String servicePackageName;

    /**
     * 服务包类型
     */
    @JsonView({View.class, RedisView.class})
    @NotNull(groups = ValidationGroup.OnCreate.class, message = "不合法的值,范围【1:标准,2:二开】")
    @Min(value = 1, groups = ValidationGroup.OnCreate.class, message = "不合法的值,范围【1:标准,2:二开】")
    @Max(value = 2, groups = ValidationGroup.OnCreate.class, message = "不合法的值,范围【1:标准,2:二开】")
    protected Integer type;

    /**
     * 服务包描述
     */
    @JsonView({View.class, RedisView.class})
    protected String servicePackageDesc;

    /**
     * 启用状态  1:启用, 0:注销
     */
    @JsonView({View.class, RedisView.class})
    @Range(max = 1L, min = 0L)
    protected Integer status;

    /**
     * 创建人id
     */
    @JsonView({View.class, RedisView.class})
    protected String createrId;

    /**
     * 创建人名称
     */
    @JsonView({View.class, RedisView.class})
    protected String createrName;

    /**
     * 创建时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date createTime;

    /**
     * 更新人id
     */
    @JsonView({View.class, RedisView.class})
    protected String updaterId;

    /**
     * 更新人名称
     */
    @JsonView({View.class, RedisView.class})
    protected String updaterName;

    /**
     * 更新时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date updateTime;

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ServicePackageDto that = (ServicePackageDto) o;
        if (this.servicePackageId != null && Objects.equals(this.servicePackageId, that.servicePackageId)) {
            return true;
        }
        return Objects.equals(this.servicePackageCode, that.servicePackageCode);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.servicePackageCode);
    }
}
