package com.xforceplus.domain.user;


import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.tenant.security.core.domain.AuthorizedUser;
import com.xforceplus.tenant.security.core.domain.IOperator;
import com.xforceplus.tenant.security.core.domain.view.UserView;
import com.xforceplus.tenant.security.token.domain.view.RedisView;
import com.xforceplus.utils.DateUtils;
import io.geewit.core.jackson.view.View;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.FieldNameConstants;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.format.annotation.DateTimeFormat;

import java.util.Date;
import java.util.Set;


/**
 * 用户
 * @author geewit
 */
@Schema(name = "用户Dto")
@FieldNameConstants
@Setter
@Getter
@ToString
public class UserDto<O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> extends AuthorizedUser<O, R> implements IOperator {
    /**
     * 用户编码 预留给业务系统使用
     */
    @JsonView({View.class, RedisView.class})
    protected String userNumber;

    /**
     * 联系电话
     */
    @JsonView({View.class, RedisView.class})
    protected String userPhone;

    /**
     * 性别
     */
    @JsonView({View.class, RedisView.class})
    protected Integer userSex;

    @JsonView({View.class, RedisView.class})
    protected String userEmailAddr;

    @JsonView({View.class, RedisView.class})
    protected String userIdCard;

    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    @JsonView({View.class, RedisView.class})
    protected Date userPeriodTime;

    @JsonView({View.class, RedisView.class})
    protected String userWorkTel;

    @JsonView({View.class, RedisView.class})
    protected Integer activeStatus;

    /**
     * 启用状态  1:启用, 0:注销
     */
    @JsonView({View.class, RedisView.class})
    protected Integer status;

    /**
     * 创建时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date createTime;

    /**
     * 创建人id
     */
    @JsonView({View.class, RedisView.class})
    protected String createrId;

    /**
     * 创建人名称
     */
    @JsonView({View.class, RedisView.class})
    protected String createrName;

    /**
     * 更新人id
     */
    @JsonView({View.class, RedisView.class})
    protected String updaterId;

    /**
     * 更新人名称
     */
    @JsonView({View.class, RedisView.class})
    protected String updaterName;

    /**
     * 更新时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date updateTime;

    /**
     * 过期日期
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd")
    @JsonFormat(pattern = "yyyy-MM-dd", timezone = "GMT+8")
    protected Date expiredDate;

    @JsonView({View.class, RedisView.class})
    protected String contactAddr;

    /**
     * 业务扩展属性
     */
    @JsonView({View.class, RedisView.class})
    protected String businessExtensionAttribute;
    /**
     * 发票类型  c:普票; s:专票; cs:专票和普票; ce:普电票
     */
    @JsonView({View.class, RedisView.class})
    protected String invoiceType;
    /**
     * 打印设备 多个打印设备值用','隔开
     */
    @JsonView({View.class, RedisView.class})
    protected String printingEquipment;
    /**
     * 开票终端 多个开票终端用','分隔开
     */
    @JsonView({View.class, RedisView.class})
    protected String ticketOpeningTerminal;

    /**
     * 账号信息
     */
    @Schema(implementation = AccountDto.class)
    @JsonView({View.class, RedisView.class})
    protected A account;

    /**
     * 过滤组织parentIds集合
     */
    protected Set<String> filterParentIdSet;

    /**
     * 来源类型 1:内部（默认） 2:外部
     */
    @JsonView({View.class, RedisView.class})
    protected Integer sourceType;

    /**
     * 组织身份  1:普通, 2:管理员
     */
    @JsonView({View.class})
    protected Integer orgRelType;

    /**
     * 当前用户是否是指定组织节点管理员
     */
    @JsonView({View.Tree.class, View.List.class, RedisView.class})
    protected Boolean isOrgAdmin;

    /**
     * 当前用户是否关联全部组织节点
     */
    @JsonView({View.Tree.class, View.List.class})
    protected Boolean hasFullOrgs;

    /**
     * 是否是唯一一个租户管理员
     */
    @JsonView({UserView.UserInfo.class, View.Tree.class, View.List.class})
    protected Boolean isLastTenantAdmin;

    /**
     * 是否租户管理员
     */
    @JsonView({UserView.UserInfo.class, View.Tree.class, View.List.class})
    protected Boolean isTenantAdmin;

    /**
     * account.username
     */
    @JsonView({View.class})
    protected String accountUserName;

    /**
     * account.email
     */
    @JsonView({View.class})
    protected String accountEmail;

    /**
     * account.telPhone
     */
    @JsonView({View.class})
    protected String accountTelPhone;

    /**
     * 最后登录时间
     */
    @JsonView({View.class, RedisView.class})
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    protected Date lastLoginTime;

    public void setBusinessExtensionAttribute(String businessExtensionAttribute) {
        this.businessExtensionAttribute = ObjectUtils.defaultIfNull(businessExtensionAttribute, StringUtils.EMPTY);
    }

    public void setInvoiceType(String invoiceType) {
        this.invoiceType = ObjectUtils.defaultIfNull(invoiceType, StringUtils.EMPTY);
    }

    public void setPrintingEquipment(String printingEquipment) {
        this.printingEquipment = ObjectUtils.defaultIfNull(printingEquipment, StringUtils.EMPTY);
    }

    public void setTicketOpeningTerminal(String ticketOpeningTerminal) {
        this.ticketOpeningTerminal = ObjectUtils.defaultIfNull(ticketOpeningTerminal, StringUtils.EMPTY);
    }

    @SuppressWarnings("all")
    @Override
    @JsonView({UserView.UserInfo.class})
    public Set<O> getOrgs() {
        return currentOrgs;
    }

    @Override
    @JsonView({UserView.UserInfo.class, RedisView.class})
    public Set<O> getCurrentOrgs() {
        return currentOrgs;
    }

    @JsonView({View.class, RedisView.class})
    public A getAccount() {
        return account;
    }

    /**
     * @return 是否过期
     */
    @JsonView({View.class})
    public boolean isExpired() {
        if (this.expiredDate == null) {
            return false;
        }
        return DateUtils.beforeNow(this.expiredDate, false);
    }
}
