/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: DateUtils.java   2020-10-20 10-49-08
 * Author: Evan
 */
package com.xforceplus.utils;


import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;


/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 10-49-08
 * @since V1.0
 */
public class DateUtils {
    private static final String SPACE = " ";

    /**
     * DateTimeFormatter  缓存
     */
    private static final ConcurrentMap<String, DateTimeFormatter> FORMATTER_CACHE = new ConcurrentHashMap<>();
    /**
     * 缓存大小为：500
     */
    private static final int PATTERN_CACHE_SIZE = 500;


    /**
     * Date转换为格式化时间
     *
     * @param date    date
     * @param pattern 格式
     * @return
     */
    public static String format(Date date, String pattern) {
        return format(LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault()), pattern);
    }

    /**
     * localDateTime转换为格式化时间
     *
     * @param localDateTime localDateTime
     * @param pattern       格式
     * @return
     */
    public static String format(LocalDateTime localDateTime, String pattern) {
        DateTimeFormatter formatter = createCacheFormatter(pattern);
        return localDateTime.format(formatter);
    }

    /**
     * 格式化字符串转为Date
     *
     * @param time    格式化时间
     * @param pattern 格式
     * @return
     */
    public static Date parseDate(String time, String pattern) {
        return Date.from(parseLocalDateTime(time, pattern).atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 获取时间结束时间
     *
     * @param date
     * @return
     */
    public static Date getDateByDayEnd(Date date) {
        if (null == date) {
            return null;
        }

        return parseDate(format(date, DatePattern.YYYY_MM_DD) + SPACE + DatePattern.TIME_END_STR, DatePattern.YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 获取时间当天的开始时间
     *
     * @param date
     * @return Date
     */
    public static Date getDateByDayBegin(Date date) {
        if (null == date) {
            return null;
        }
        return parseDate(format(date, DatePattern.YYYY_MM_DD) + SPACE + DatePattern.TIME_BEGIN_STR, DatePattern.YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 格式化字符串转为LocalDateTime
     *
     * @param time    格式化时间
     * @param pattern 格式
     * @return
     */
    public static LocalDateTime parseLocalDateTime(String time, String pattern) {
        DateTimeFormatter formatter = createCacheFormatter(pattern);
        return LocalDateTime.parse(time, formatter);
    }

    /**
     * 在缓存中创建DateTimeFormatter
     *
     * @param pattern 格式
     * @return
     */
    private static DateTimeFormatter createCacheFormatter(String pattern) {
        if (pattern == null || pattern.length() == 0) {
            throw new IllegalArgumentException("Invalid pattern specification");
        }
        DateTimeFormatter formatter = FORMATTER_CACHE.get(pattern);
        if (formatter == null) {
            if (FORMATTER_CACHE.size() < PATTERN_CACHE_SIZE) {
                formatter = DateTimeFormatter.ofPattern(pattern);
                DateTimeFormatter oldFormatter = FORMATTER_CACHE.putIfAbsent(pattern, formatter);
                if (oldFormatter != null) {
                    formatter = oldFormatter;
                }
            }
        }
        return formatter;
    }

    /**
     * 获取自定义时间
     *
     * @param hour
     * @param minute
     * @param second
     * @return
     */
    public static long getCustomTime(int hour, int minute, int second) {
        Calendar time = Calendar.getInstance();
        time.set(Calendar.HOUR_OF_DAY, hour);
        time.set(Calendar.MINUTE, minute);
        time.set(Calendar.SECOND, second);
        time.set(Calendar.MILLISECOND, 0);
        return time.getTimeInMillis();
    }

    public static boolean before(Date aDate, Date bDate, boolean includeSameDate) {
        Calendar aCal = org.apache.commons.lang3.time.DateUtils.toCalendar(aDate);
        Calendar bCal = org.apache.commons.lang3.time.DateUtils.toCalendar(bDate);
        return before(aCal, bCal, includeSameDate);
    }

    public static boolean after(Date aDate, Date bDate, boolean includeSameDate) {
        Calendar aCal = org.apache.commons.lang3.time.DateUtils.toCalendar(aDate);
        Calendar bCal = org.apache.commons.lang3.time.DateUtils.toCalendar(bDate);
        return after(aCal, bCal, includeSameDate);
    }
    public static boolean before(Calendar aCal, Calendar bCal, boolean includeSameDate) {
        int compared = calendarCompareTo(aCal, bCal);
        if(compared > 0) {
            return false;
        } else if (compared == 0) {
            return includeSameDate;
        } else {
            return true;
        }
    }

    public static boolean after(Calendar aCal, Calendar bCal, boolean includeSameDate) {
        int compared = calendarCompareTo(aCal, bCal);
        if(compared > 0) {
            return true;
        } else if (compared == 0) {
            return includeSameDate;
        } else {
            return false;
        }
    }

    public static boolean beforeNow(Date date, boolean includeSameDate) {
        Calendar nowCal = Calendar.getInstance();
        Calendar cal = org.apache.commons.lang3.time.DateUtils.toCalendar(date);
        return before(cal, nowCal, includeSameDate);
    }

    public static boolean afterNow(Date date, boolean includeSameDate) {
        Calendar nowCal = Calendar.getInstance();
        Calendar cal = org.apache.commons.lang3.time.DateUtils.toCalendar(date);
        return after(cal, nowCal, includeSameDate);
    }

    public static int calendarCompareTo(Calendar aCal, Calendar bCal) {
        aCal.set(Calendar.HOUR_OF_DAY, 0);
        aCal.set(Calendar.MINUTE, 0);
        aCal.set(Calendar.SECOND, 0);
        aCal.set(Calendar.MILLISECOND, 0);
        bCal.set(Calendar.HOUR_OF_DAY, 0);
        bCal.set(Calendar.MINUTE, 0);
        bCal.set(Calendar.SECOND, 0);
        bCal.set(Calendar.MILLISECOND, 0);
        return aCal.compareTo(bCal);
    }
}
