package com.xforceplus.utils;

import com.google.common.base.Splitter;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.tenant.security.core.domain.OrgType;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

/**
 * 生成树
 *
 * @author geewit
 */
@SuppressWarnings("all")
public class OrgUtils {
    private final static Logger logger = LoggerFactory.getLogger(OrgUtils.class);

    private static final String REGEX_SEPERATOR = "|";
    private static final String REGEX_SEPERATOR_PREFIX = "[";
    private static final String REGEX_SEPERATOR_SURFIX = "]";

    private static final String SPLIT_OF_ORG_PARENT_IDS = Arrays.stream(OrgType.values())
            .map(OrgType::separator)
            .collect(Collectors.joining(REGEX_SEPERATOR, REGEX_SEPERATOR_PREFIX, REGEX_SEPERATOR_SURFIX));

    private static final String SPLITTER_OF_ORG_PARENT_IDS = Arrays.stream(OrgType.values())
            .map(OrgType::separator)
            .collect(Collectors.joining());
    private static final Pattern SPLIT_PATTERN_OF_ORG_PARENT_IDS = Pattern.compile(SPLIT_OF_ORG_PARENT_IDS);

    private static final char[] SEPARATOR_CHARS = new char[OrgType.values().length];

    static {
        for (int i = 0; i < OrgType.values().length; i++) {
            SEPARATOR_CHARS[i] = OrgType.values()[i].separator().charAt(0);
        }
    }

    private static final Map<OrgType, Pattern> SPLIT_PATTERN_OF_ORG_PARENT_IDS_MAP = Arrays.stream(OrgType.values())
            .collect(Collectors.toMap(Function.identity(), orgType -> Pattern.compile("(\\d+)" + REGEX_SEPERATOR_PREFIX + orgType.separator() + REGEX_SEPERATOR_SURFIX), (oldKey, newKey) -> oldKey));

    public static boolean containsInParentIds(String parentIds, String orgId) {
        if (StringUtils.isBlank(parentIds)) {
            return false;
        }

        int start = 0;
        int orgIdLength = orgId.length();
        while (start >= 0) {
            int index = indexOf(parentIds, orgId, start);
            if (index < 0) {
                return false;
            }
            start = index + orgIdLength;
            boolean foundLastSeparator = false;
            if (index == 0) {
                foundLastSeparator = true;
            } else {
                char lastSeparatorChar = parentIds.charAt(index - 1);
                for (char separator : SEPARATOR_CHARS) {
                    if (lastSeparatorChar == separator) {
                        foundLastSeparator = true;
                        break;
                    }
                }
            }
            if (foundLastSeparator) {
                char separatorChar = parentIds.charAt(start);
                boolean found = false;
                for (char separator : SEPARATOR_CHARS) {
                    if (separatorChar == separator) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    static int indexOf(final CharSequence cs, final CharSequence searchChar, final int start) {
        if (cs instanceof String) {
            return ((String) cs).indexOf(searchChar.toString(), start);
        }
        if (cs instanceof StringBuilder) {
            return ((StringBuilder) cs).indexOf(searchChar.toString(), start);
        }
        if (cs instanceof StringBuffer) {
            return ((StringBuffer) cs).indexOf(searchChar.toString(), start);
        }
        return cs.toString().indexOf(searchChar.toString(), start);
    }


    /**
     * 根据parentIds 查询id集合
     *
     * @param parentIds
     * @return
     */
    public static Set<Long> findOrgIdInParentIds(String parentIds) {
        if (StringUtils.isBlank(parentIds)) {
            return Collections.emptySet();
        }
        Set<Long> orgIds = Splitter.on(SPLIT_PATTERN_OF_ORG_PARENT_IDS).omitEmptyStrings().trimResults()
                .splitToStream(parentIds).filter(NumberUtils::isDigits)
                .map(v -> {
                    try {
                        long id = Long.parseLong(v);
                        if (id > 0) {
                            return id;
                        } else {
                            return null;
                        }
                    } catch (NumberFormatException e) {
                        logger.info("{}, value: {}", e.getMessage(), v);
                        return null;
                    }
                }).filter(Objects::nonNull)
                .collect(Collectors.toSet());
        return orgIds;
    }

    public static Set<Long> findOrgIdsInParentIdsByOrgType(String parentIds, OrgType orgType, boolean findFirst) {
        logger.debug("parentIds = " + parentIds);
        if (StringUtils.isBlank(parentIds)) {
            return Collections.emptySet();
        }
        Matcher matcher = SPLIT_PATTERN_OF_ORG_PARENT_IDS_MAP.get(orgType).matcher(parentIds);
        Set<Long> orgIds = new HashSet<>();
        while (matcher.find()) {
            String group = matcher.group(1);
            try {
                long orgId = Long.parseLong(group);
                if (logger.isDebugEnabled()) {
                    logger.debug("orgId = " + orgId);
                }
                if (orgId <= 0) {
                    continue;
                }
                orgIds.add(orgId);
                if (findFirst && orgId > 0) {
                    break;
                }
            } catch (NumberFormatException e) {
                logger.info("{}, value: {}", e.getMessage(), group);
            }
        }
        return orgIds;
    }

    public static Set<Long> findCompanyOrgIdsInParentIds(String parentIds) {
        Set<Long> orgIds = OrgUtils.findOrgIdsInParentIdsByOrgType(parentIds, OrgType.COMPANY, false);
        return orgIds;
    }

    public static Long findRootIdInParentIds(String parentIds) {
        Set<Long> orgIds = OrgUtils.findOrgIdsInParentIdsByOrgType(parentIds, OrgType.GROUP, true);
        return orgIds.stream().findFirst().orElseGet(null);
    }

    /**
     * 压缩parentIds集合减少sql次数
     * ['10001#', '10001#10002/', '10001#10003/', '10001#10004/', '20001#20002/', '20001#20003/'] -> ['10001#', '20001#20002/', '20001#20003/']
     * @param parentIdsCollection
     * @return
     */
    public static Set<String> compressParentIdsCollection(Collection<String> parentIdsCollection) {
        if (parentIdsCollection == null) {
            return Collections.emptySet();
        }
        if (parentIdsCollection.isEmpty()) {
            if (parentIdsCollection instanceof Set) {
                return (Set<String>) parentIdsCollection;
            } else {
                return Collections.emptySet();
            }
        }
        Stack<String> parentIdsStack = new Stack<>();
        //region 压缩parentIds集合减少sql次数
        parentIdsCollection.stream()
                .filter(StringUtils::isNotBlank)
                .sorted(String::compareTo).forEach(currentParentIds -> {
                    if (parentIdsStack.isEmpty()) {
                        parentIdsStack.push(currentParentIds);
                    } else {
                        String parentIds = parentIdsStack.pop();
                        if (!parentIds.equals(currentParentIds)) {
                            if (parentIds.startsWith(currentParentIds)) {
                                parentIdsStack.push(currentParentIds);
                            } else if (currentParentIds.startsWith(parentIds)) {
                                parentIdsStack.push(parentIds);
                            } else {
                                parentIdsStack.push(parentIds);
                                parentIdsStack.push(currentParentIds);
                            }
                        } else {
                            parentIdsStack.push(parentIds);
                        }
                    }
                });
        return parentIdsStack.stream().collect(Collectors.toSet());
    }

    /**
     * 过滤orgIds中不属于parentIdsCollection中的集合
     *
     * @param orgIds
     * @param parentIdsCollection
     * @return
     */
    public static Set<Long> excludeInParentIds(Collection<Long> orgIds, Collection<String> parentIdsCollection) {
        Set<Long> filteredOrgIds;
        if (orgIds == null) {
            return Collections.emptySet();
        } else {
            if (orgIds instanceof Set) {
                filteredOrgIds = (Set<Long>) orgIds;
            } else {
                filteredOrgIds = new HashSet<>(orgIds);
            }
        }
        if (CollectionUtils.isEmpty(filteredOrgIds)) {
            return filteredOrgIds;
        }
        if (CollectionUtils.isEmpty(parentIdsCollection)) {
            return filteredOrgIds;
        }
        for (String parentIds : parentIdsCollection) {
            String filterParentIdsWithoutLastSperator = StringUtils.substring(parentIds, 0, parentIds.length() - 1);
            filteredOrgIds = filteredOrgIds.stream()
                    .filter(id -> !StringUtils.endsWith(filterParentIdsWithoutLastSperator, String.valueOf(id)))
                    .collect(Collectors.toSet());
        }
        return filteredOrgIds;
    }

    /**
     * 压缩 parentIds 集合 将 重复且包含的parentIds去掉
     *
     * @param parentIdsCollection
     * @param orgIds
     * @return
     */
    public static Pair<Set<String>, Set<Long>> compressParentIdsCollectionAndFilterOrgIds(Collection<String> parentIdsCollection, Collection<Long> orgIds) {
        Set<Long> filteredOrgIds;
        if (orgIds == null) {
            filteredOrgIds = Collections.emptySet();
        } else {
            if (orgIds instanceof Set) {
                filteredOrgIds = (Set<Long>) orgIds;
            } else {
                filteredOrgIds = new HashSet<>(orgIds);
            }
        }

        Stack<String> parentIdsStack = new Stack<>();
        final Set<Long> finalFilteredOrgIds = filteredOrgIds;
        //region 压缩parentIds集合减少sql次数
        parentIdsCollection.stream()
                .filter(StringUtils::isNotBlank)
                .sorted(String::compareTo)
                .forEach(currentParentIds -> {
                    if (parentIdsStack.isEmpty()) {
                        parentIdsStack.push(currentParentIds);
                    } else {
                        String parentIds = parentIdsStack.pop();
                        OrgUtils.filterOrgIdsInParentIds(finalFilteredOrgIds, currentParentIds);
                        if (!parentIds.equals(currentParentIds)) {
                            if (parentIds.startsWith(currentParentIds)) {
                                parentIdsStack.push(currentParentIds);
                            } else if (currentParentIds.startsWith(parentIds)) {
                                parentIdsStack.push(parentIds);
                            } else {
                                parentIdsStack.push(parentIds);
                                parentIdsStack.push(currentParentIds);
                            }
                        } else {
                            parentIdsStack.push(parentIds);
                        }
                    }
                });
        Set<String> parentIdsSet = parentIdsStack.stream().collect(Collectors.toSet());
        //endregion
        return Pair.of(parentIdsSet, filteredOrgIds);
    }

    /**
     * 过滤在 like 条件中和 equal 重复的 includeOrgIds, includeOrgIds只保留于like条件不重合的条目
     * @param orgIds
     * @param parentIds
     */
    public static void filterOrgIdsInParentIds(Set<Long> orgIds, String parentIds) {
        if (CollectionUtils.isNotEmpty(orgIds) && StringUtils.isNotBlank(parentIds)) {
            //region 去除 parentIds最后的'OrgType分割'符号
            String finalParentIds = StringUtils.substring(parentIds, 0, parentIds.length() - 1);
            //endregion
            Set<Long> filteredOrgIds = orgIds.stream()
                    .filter(id -> !StringUtils.endsWith(finalParentIds, String.valueOf(id)))
                    .collect(Collectors.toSet());
            orgIds.clear();
            orgIds.addAll(filteredOrgIds);
        }
    }

    /**
     * 过滤指定租户下有权限的组织parentIds集合
     * @param tenantId 指定租户id
     * @param modules 指定隔离模块 (orgCode)
     * @param orgs 指定租户下的所有组织
     * @return
     * @param <O>
     */
    public static <O extends OrgDto<O>> Set<String> filterParentIdsByModules(long tenantId, String modules, List<O> orgs) {
        String[] orgCodeArray = StringUtils.split(modules, ",");
        if (ArrayUtils.isEmpty(orgCodeArray)) {
            return Collections.emptySet();
        }
        Set<String> orgCodes = Stream.of(orgCodeArray).filter(Objects::nonNull).collect(toSet());
        if (orgCodes.isEmpty()) {
            return Collections.emptySet();
        }
        List<String> filteringParentIdsList = orgs.stream()
                .filter(o -> o.getTenantId() != null && tenantId == o.getTenantId() && orgCodes.stream()
                        .anyMatch(code -> code.equals(o.getOrgCode()))
                )
                .map(O::getParentIds)
                .collect(Collectors.toList());
        return OrgUtils.compressParentIdsCollection(filteringParentIdsList);
    }

    /**
     * 将parentIds集合和orgId集合转换成Map
     * @param parentIdsSet
     * @param orgIds
     * @return
     */
    public static Map<Long, Set<Long>> parentIdsMapOrgIds(Set<String> parentIdsSet, Set<Long> orgIds) {
        return parentIdsSet.stream()
                .collect(Collectors.toMap(parentIds ->
                                orgIds.stream()
                                        .filter(orgId -> StringUtils.endsWith(StringUtils.substring(parentIds, 0, parentIds.length() - 1), String.valueOf(orgId))).findFirst().orElse(null),
                        parentIds -> {
                    return OrgUtils.findOrgIdInParentIds(parentIds).stream().filter(orgId -> !orgIds.contains(orgId)).collect(toSet());
                }, (oldKey, newKey) -> newKey)).entrySet().stream().filter(entry -> entry.getKey() != null).collect(toMap(Map.Entry::getKey, Map.Entry::getValue, (oldKey, newKey) -> oldKey));
    }
}
