package com.xforceplus.redis.lock.interceptor;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.xforceplus.redis.lock.RedisLock;
import com.xforceplus.redis.lock.manager.RedisManager;
import io.geewit.web.utils.JsonUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

/**
 * redis锁拦截器实现
 *
 * @author MAZHEN
 */
@SuppressWarnings("all")
@Aspect
@Component
@ConditionalOnBean(RedisTemplate.class)
public class RedisLockInterceptor {
    private static final Logger logger = LoggerFactory.getLogger(RedisLockInterceptor.class);
    private static final String PROPERTY_KEY = "key";
    private static final Integer MAX_RETRY_COUNT = 10;
    private static final String LOCK_PRE_FIX = "lockPreFix";
    private static final String TIME_OUT = "timeOut";


    @Autowired
    private RedisManager redisManager;

    public static Object getFieldValueByObject(Object object, String targetFieldName) throws Exception {
        // 获取该对象的Class
        Class objClass = object.getClass();
        // 获取所有的属性数组
        Field[] fields = objClass.getDeclaredFields();
        for (Field field : fields) {
            // 属性名称
            String currentFieldName = "";
            // 获取属性上面的注解 import com.fasterxml.jackson.annotation.JsonProperty;

//              举例：
//              @JsonProperty("di_ren_jie")
//             private String diRenJie;

            boolean hasJsonProperty = field.isAnnotationPresent(JsonProperty.class);
            if (hasJsonProperty) {
                currentFieldName = field.getAnnotation(JsonProperty.class).value();
            } else {
                currentFieldName = field.getName();
            }
            if (currentFieldName.equals(targetFieldName)) {
                //通过反射拿到该属性在此对象中的值(也可能是个对象)
                return field.get(object);
            }
        }
        return null;
    }

    @Pointcut("@annotation( com.xforceplus.redis.lock.RedisLock)")
    public void redisLockAspect() {
    }

    @Around("redisLockAspect()")
    public Object lockAroundAction(ProceedingJoinPoint proceeding) throws Throwable {
        Object[] args = proceeding.getArgs();
        //获取redis锁
        Map<String, Object> getLockResult = this.getLock(proceeding, 0, System.currentTimeMillis());
        logger.info("获取锁的结果为{},{}", getLockResult.get("code"), getLockResult.get("msg"));

        logger.info("开始处理");
        Object reps = proceeding.proceed();
        logger.info("结果为{}", JsonUtils.toJson(reps));
        return reps;

    }

    /**
     * 获取锁
     *
     * @param proceeding
     * @return
     */
    private Map<String, Object> getLock(ProceedingJoinPoint proceeding, int count, long currentTime) {
        //获取注解中的参数
        Map<String, Object> annotationArgs = this.getAnnotationArgs(proceeding);
        String lockPrefix = (String) annotationArgs.get(LOCK_PRE_FIX);
        String property = (String) annotationArgs.get(PROPERTY_KEY);
        long expire = (long) annotationArgs.get(TIME_OUT);
        String key = this.getFirstArg(proceeding, property);
        if (StringUtils.isEmpty(lockPrefix) || StringUtils.isEmpty(key)) {
            return this.argErrResult("锁前缀或业务参数不能为空");
        }
        String lockName = lockPrefix + "_" + key;
        logger.info("获得所得key为{}", lockName);
        if (redisManager.lock(lockPrefix, key, expire)) {
            return this.buildSuccessResult();
        } else {
            //获取锁失败,为防止其它线程正在设置过时时间时误删，添加第一个条件
            if ((System.currentTimeMillis() - currentTime > 5000)
                    && (redisManager.ttl(lockName) < 0
                    || System.currentTimeMillis() - currentTime > expire)) {
                //强制删除锁，并尝试再次获取锁
                redisManager.delete(lockName);
                if (count < MAX_RETRY_COUNT) {
                    return getLock(proceeding, count++, currentTime);
                }
            }
            return this.buildGetLockErrorResult("请重试！！！");
        }
    }

    /**
     * 获取锁参数
     *
     * @param proceeding
     * @return
     */
    private Map<String, Object> getAnnotationArgs(ProceedingJoinPoint proceeding) {
        Class target = proceeding.getTarget().getClass();
        Method[] methods = target.getMethods();
        String methodName = proceeding.getSignature().getName();
        for (Method method : methods) {
            if (method.getName().equals(methodName)) {
                Map<String, Object> result = new HashMap<>();
                RedisLock redisLock = method.getAnnotation(RedisLock.class);
                result.put(LOCK_PRE_FIX, redisLock.lockPrefix());
                result.put(TIME_OUT, redisLock.timeUnit().toSeconds(redisLock.timeOut()));
                result.put(PROPERTY_KEY, redisLock.key());
                return result;
            }
        }
        return new HashMap<>(0);
    }

    /**
     * 获取第一个String类型的参数为锁的业务参数
     *
     * @param proceeding
     * @return
     */

    public String getFirstArg(ProceedingJoinPoint proceeding, String property) {
        Object[] args = proceeding.getArgs();

        if (args != null && args.length > 0) {

            if (StringUtils.hasText(property)) {
                try {
                    Object jsonObject = getFieldValueByObject(args[0], property);

                    if (null == jsonObject) {
                        logger.error("获取指定属性值失败{},{}", JsonUtils.toJson(args[0]), property);
                        return proceeding.getClass().getName();
                    }
                    return jsonObject.toString();
                } catch (Exception e) {
                    logger.error("获取指定属性值失败{},{}", JsonUtils.toJson(args[0]), property);
                    logger.error("获取指定属性值失败原因:" + e.getMessage(), e);
                    return proceeding.getClass().getName();
                }
            }
            for (Object object : args) {
                String type = object.getClass().getName();
                if ("java.lang.String".equals(type)) {
                    return (String) object;
                }
            }
        }

        return proceeding.getClass().getName();
    }

    public Map<String, Object> argErrResult(String mes) {
        Map<String, Object> result = new HashMap<String, Object>(2);
        result.put("code", "9");
        result.put("msg", mes);
        return result;
    }

    public Map<String, Object> buildGetLockErrorResult(String mes) {
        Map<String, Object> result = new HashMap<String, Object>(2);
        result.put("code", "9");
        result.put("msg", mes);
        return result;
    }

    public Map<String, Object> buildSuccessResult() {
        Map<String, Object> result = new HashMap<String, Object>(2);
        result.put("code", "1");
        result.put("msg", "处理成功");
        return result;
    }
}
