package com.xforceplus.advice;

import com.xforceplus.tenant.security.core.context.ClientInfoHolder;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.utils.IpUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.MethodParameter;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.RequestBodyAdviceAdapter;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.lang.reflect.Type;
import java.util.Arrays;

/**
 * <p>
 * 这个advice仅支持参数为@RequestBody的接口日志，一般为POST、PUT、PATCH等类型的请求
 * 对于Get请求及文件上传，不会走到这个逻辑，通过tenant-security提供的interceptor进行拦截记录简单日志
 * </p>
 *
 * @author chenpengpeng@xforceplus.com
 * @date 2021/6/24
 **/
@Slf4j
@ControllerAdvice
public class XforceRequestBodyAdviceAdapter extends RequestBodyAdviceAdapter {
    @Autowired
    HttpServletRequest httpServletRequest;

    private static final String PASSWORD = "password";

    /**
     * Invoked first to determine if this interceptor applies.
     *
     * @param methodParameter the method parameter
     * @param targetType      the target type, not necessarily the same as the method
     *                        parameter type, e.g. for {@code HttpEntity<String>}.
     * @param converterType   the selected converter type
     * @return whether this interceptor should be invoked or not
     */
    @Override
    public boolean supports(MethodParameter methodParameter, Type targetType, Class<? extends HttpMessageConverter<?>> converterType) {
        return true;
    }

    /**
     * The default implementation returns the InputMessage that was passed in.
     *
     * @param inputMessage
     * @param parameter
     * @param targetType
     * @param converterType
     */
    @Override
    public HttpInputMessage beforeBodyRead(HttpInputMessage inputMessage, MethodParameter parameter, Type targetType, Class<? extends HttpMessageConverter<?>> converterType) throws IOException {
        return super.beforeBodyRead(inputMessage, parameter, targetType, converterType);
    }

    /**
     * The default implementation returns the body that was passed in.
     *
     * @param body
     * @param inputMessage
     * @param parameter
     * @param targetType
     * @param converterType
     */
    @Override
    public Object afterBodyRead(Object body, HttpInputMessage inputMessage, MethodParameter parameter, Type targetType, Class<? extends HttpMessageConverter<?>> converterType) {
        try {
            logRequest(httpServletRequest, body);
        } catch (Exception exception) {
            log.error("error while logging request info:{}", exception.getMessage());
        }
        return super.afterBodyRead(body, inputMessage, parameter, targetType, converterType);
    }

    /**
     * The default implementation returns the body that was passed in.
     *
     * @param body
     * @param inputMessage
     * @param parameter
     * @param targetType
     * @param converterType
     */
    @Override
    public Object handleEmptyBody(Object body, HttpInputMessage inputMessage, MethodParameter parameter, Type targetType, Class<? extends HttpMessageConverter<?>> converterType) {
        return super.handleEmptyBody(body, inputMessage, parameter, targetType, converterType);
    }

    private void logRequest(HttpServletRequest request, Object body) {
        String remoteIp = IpUtils.getIp(request);
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("REQUEST INFO:");
        stringBuilder.append("path = [")
                .append(request.getRequestURI())
                .append("(")
                .append(request.getMethod())
                .append(")")
                .append("];");
        stringBuilder.append("remoteIp=[")
                .append(remoteIp)
                .append("]")
                .append(";");
        String queryString = request.getQueryString();
        if (queryString != null) {
            queryString = queryString.replace("&", ";");
            queryString = this.filterSensitiveLog(queryString);
            stringBuilder.append("queryString = [")
                    .append(queryString)
                    .append("];");
        }
        //必须启用tenantSecurity的interceptor才能用这个方法拿用户上下文信息
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        if (authorizedUser != null) {
            stringBuilder.append("userId = [")
                    .append(authorizedUser.getId())
                    .append("];")
                    .append("tenantId = [")
                    .append(authorizedUser.getTenantId())
                    .append("];");
        } else {
            String clientId = ClientInfoHolder.get();
            if (clientId != null) {
                stringBuilder.append("clientId=[")
                        .append(clientId)
                        .append("];");
            }
        }
        if (body != null) {
            stringBuilder.append("body = [")
                    .append(body)
                    .append("];");
        }
        log.info(stringBuilder.toString());
    }

    /**
     * 将queryString中的敏感信息屏蔽掉
     *
     * @param source queryString
     * @return 去掉敏感信息的字符串
     */
    private String filterSensitiveLog(String source) {
        final int keyLength = 2;
        if (StringUtils.isBlank(source)) {
            return null;
        } else {
            if (!source.contains(PASSWORD)) {
                return source;
            } else {
                StringBuilder sb = new StringBuilder();
                String[] elements = source.split(";");
                Arrays.stream(elements).forEach(item -> {
                    String[] kv = item.split("=");
                    if (!kv[0].equals(PASSWORD)) {
                        sb.append(item).append(";");
                    } else {
                        sb.append("pwdlength")
                                .append("=");
                        //仅有key没有value的情况
                        if (kv.length < keyLength) {
                            sb.append(0).append(";");
                        } else {
                            //value为“”的情况
                            if (StringUtils.isNotBlank(kv[1])) {
                                sb.append(kv[1].length());
                            } else {
                                sb.append(0);
                            }
                            sb.append(";");
                        }
                    }
                });
                return sb.toString();
            }
        }
    }

}
