package com.xforceplus.business.account.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.account.AccountApi;
import com.xforceplus.api.model.AccountModel;
import com.xforceplus.api.model.AccountModel.AccountPrivacyInfo;
import com.xforceplus.api.model.AccountModel.Request.*;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.business.account.service.AccountPrivacyService;
import com.xforceplus.business.account.service.AccountService;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.account.AccountExceptionDto;
import com.xforceplus.domain.account.AccountType;
import com.xforceplus.dto.user.AccountUserDTO;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.User;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.core.jackson.view.View;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.util.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.WebRequest;
import springfox.documentation.annotations.ApiIgnore;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * @author geewit
 */

@SuppressWarnings("all")
@Api(value = "帐号相关接口", description = "帐号相关接口")
@Validated
@Controller
public class AccountController implements AccountApi, com.xforceplus.api.current.account.AccountApi {
    private final static Logger logger = LoggerFactory.getLogger(AccountController.class);


    private final AccountService accountService;

    private final UserService userService;

    private final AccountPrivacyService accountPrivacyService;

    public AccountController(AccountService accountService, UserService userService,
                             AccountPrivacyService accountPrivacy) {
        this.accountService = accountService;
        this.userService = userService;
        this.accountPrivacyService = accountPrivacy;
    }


    @ApiOperation(value = "登录并获取用户详情")
    @Override
    public ResponseEntity<User> login(Login login, int extraInfoDimension) {
        User result = userService.login(login, extraInfoDimension);
        return ResponseEntity.ok(result);
    }

    @JsonView(View.List.class)
    @ApiOperation(value = "获取account的User列表")
    @ResponseBody
    @RequestMapping(name = "获取account的User列表", value = AccountApi.Path.ACCOUNT_USER_LIST, method = RequestMethod.GET)
    public ResponseEntity<List<AccountUserDTO>> accountUserList(@ApiParam(value = "accountId", required = true) @PathVariable("accountId") long accountId) {
        return ResponseEntity.ok(accountService.accountUserList(accountId));
    }

    /**
     * 获取帐号分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @ApiIgnore
    @JsonView(View.List.class)
    @ApiOperation(value = "获取帐号分页", notes = "通过分页参数，获取帐号分页")
    @ResponseBody
    @RequestMapping(name = "帐号分页列表查询", value = {"/api/global/accounts/query"}, method = RequestMethod.GET)
    public Page<Account> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<Account> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<Account> page = accountService.page(specification, pageable);
        return page;
    }

    @JsonView(View.List.class)
    @ApiOperation(value = "获取帐号分页", notes = "通过分页参数，获取帐号分页")
    @Override
    public ResponseEntity<Page<Account>> page(@ApiParam(value = "request") Query query,
                                              @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<Account> result = accountService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(authentication = false)
    @JsonView(View.Info.class)
    @ApiOperation(value = "通过用户名获取帐号详情", notes = "通过用户名获取帐号详情")
    public ResponseEntity<Account> queryByUsername(@PathVariable("username") String username) {
        Account result = accountService.findOneByUsername(username);
        return ResponseEntity.ok(result);
    }

    @JsonView(View.List.class)
    @ApiOperation(value = "新增帐号")
    @Override
    public ResponseEntity<Account> create(@ApiParam(value = "account", required = true) Create model) {
        Account result = accountService.create(model);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<String> register(Regist regist) {
        UserModel.Request.Create userModel = new UserModel.Request.Create();
        AccountModel.Request.Create accountModel = new Create();
        BeanUtils.copyProperties(regist, accountModel);
        userModel.setAccount(accountModel);
        if (!StringUtils.isEmpty(regist.getEmail())) {
            userModel.setUserEmailAddr(regist.getEmail());
        }
        if (!StringUtils.isEmpty(regist.getTelPhone())) {
            userModel.setUserPhone(regist.getTelPhone());
        }
        userModel.setStatus(1);
        accountModel.setType(AccountType.PHONE_EMAIL);
        UserModel.Request.SaveUserOutput userOutput = userService.register(userModel, regist.getOpenId());
        if (userOutput != null) {
            return ResponseEntity.ok("注册成功");
        } else {
            return ResponseEntity.fail("-1", "注册失败");
        }
    }

    @JsonView(View.List.class)
    @Override
    public <T extends AccountDto> ResponseEntity<List<T>> createBatch(List<Create> models) {
        List<T> resultList = new ArrayList<>();
        for (Create create : models) {
            try {
                Account result = accountService.saveOriginPassword(create);
                resultList.add((T) result);
            } catch (Exception e) {
                logger.warn(e.getMessage());
                resultList.add((T) new AccountExceptionDto(e.getMessage()));
            }
        }

        return ResponseEntity.ok(resultList);
    }

    @JsonView(View.List.class)
    @ApiOperation(value = "修改帐号")
    @Override
    public ResponseEntity<Account> update(@ApiParam(value = "accountId", required = true) long accountId,
                                          @ApiParam(value = "account", required = true) Update model) {
        Account result = accountService.update(accountId, model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "修改帐号状态")
    @Override
    public ResponseEntity<String> updateStatus(@ApiParam(value = "accountId", required = true) long accountId,
                                               @ApiParam(value = "status", required = true) int status) {
        accountService.updateStatus(accountId, status);
        return ResponseEntity.ok("修改成功");
    }

    @ApiOperation(value = "账号锁定")
    @Override
    public ResponseEntity<String> accountLock(@ApiParam(value = "accountId", required = true) long accountId) {
        accountService.updateStatus(accountId, 0);
        return ResponseEntity.ok("锁定成功");
    }

    @ApiOperation(value = "账号解锁")
    @Override
    public ResponseEntity<String> accountUnlock(@ApiParam(value = "accountId", required = true) long accountId) {
        accountService.updateStatus(accountId, 1);
        return ResponseEntity.ok("解锁成功");
    }

    @JsonView(View.Info.class)
    @ApiOperation(value = "根据id获取帐号信息")
    @Override
    public ResponseEntity<Account> info(long accountId) {
        Account result = accountService.findById(accountId);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(authentication = false)
    @ApiOperation(value = "根据username更改账户密码")
    @Override
    public ResponseEntity<String> updatePassword(ChangePassword request) {
        accountService.resetPassword(request);
        return ResponseEntity.ok("更改成功");
    }

    @AuthorizedDefinition(authentication = false)
    @ApiOperation(value = "华润根据accountId更改账户密码")
    @Override
    public ResponseEntity<String> updatePasswordCiphertext(UpdatePasswordCiphertext request) {
        accountService.updatePasswordCiphertext(request);
        return ResponseEntity.ok("更改成功");
    }

    @ApiOperation(value = "根据accountId更改账户密码")
    @Override
    public ResponseEntity<String> updatePasswordByAccountId(long accountId, ChangePasswordById request) {
        accountService.changePassword(accountId, request.getPassword());
        return ResponseEntity.ok("更改成功");
    }

    @Override
    public ResponseEntity<String> checkPassword(CheckPassword request) {
        Long accountId = UserInfoHolder.currentUser().getAccountId();
        accountService.checkPasswordByAccountId(accountId, request.getPassword());
        return ResponseEntity.ok();
    }

    @AuthorizedDefinition(authentication = false)
    @Override
    public ResponseEntity<String> sendChangePhoneCode(String phone) {
        String msgId = accountService.sendChangePhoneCode(phone);
        return ResponseEntity.ok(msgId);
    }

    @AuthorizedDefinition(authentication = false)
    @Override
    public ResponseEntity<String> sendChangeEmailCode(String email) {
        String msgId = accountService.sendChangeEmailCode(email);
        return ResponseEntity.ok(msgId);
    }

    @Override
    public ResponseEntity<String> changePhone(ChangePhone request) {
        accountService.changePhone(request);
        return ResponseEntity.ok();
    }

    @Override
    public ResponseEntity<String> changeEmail(ChangeEmail request) {
        accountService.changeEmail(request);
        return ResponseEntity.ok();
    }

    @AuthorizedDefinition(authentication = false)
    @Override
    public ResponseEntity<String> bindPhone(BindPhone request) {
        accountService.bindPhone(request);
        return ResponseEntity.ok();
    }

    @AuthorizedDefinition(authentication = false)
    @Override
    public ResponseEntity<String> bindEmail(BindEmail request) {
        accountService.bindEmail(request);
        return ResponseEntity.ok();
    }

    @AuthorizedDefinition(authentication = false)
    @Override
    public ResponseEntity<String> sendValidCode(String username) {
        String msgId = accountService.sendValidCode(username);
        return ResponseEntity.ok(msgId);
    }

    @AuthorizedDefinition(authentication = false)
    @Override
    public ResponseEntity<String> updatePwd(AccountModel.ChangePwd request) {
        accountService.updatePwd(request);
        return ResponseEntity.ok();
    }

    @Override
    @JsonView(View.Info.class)
    public ResponseEntity<Account> currentQueryByUsername(String username) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        String tenantCode = authorizedUser.getTenantCode();
        Account result = accountService.findOneByUsername(tenantCode, username);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(authentication = false)
    @Override
    public ResponseEntity<String> currentUpdatePassword(ChangePassword request) {
        accountService.resetPassword(request);
        return ResponseEntity.ok("更改成功");
    }

    @Override
    public ResponseEntity<String> unBindPhone(UnBindPhone request) {
        accountService.unBindPhone(request);
        return ResponseEntity.ok();
    }

    @Override
    public ResponseEntity<String> unBindEmail(UnBindEmail request) {
        accountService.unBindEmail(request);
        return ResponseEntity.ok();
    }

    @Override
    public ResponseEntity<User> changeTenant(long accountId, long tenantId) {
        User user = userService.changeTenantByAccount(accountId, tenantId);
        return ResponseEntity.ok(user);
    }

    @Override
    public ResponseEntity<String> currentUserUpdatePassword(AccountModel.Request.ChangeCurrentPassword request) {
        accountService.resetCurrentUserPassword(request);
        return ResponseEntity.ok();
    }

    @Override
    @JsonView(View.Info.class)
    public ResponseEntity<Account> currentCheckByUsername(String username) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(username)) {
            throw new IllegalArgumentException("用户名格式错误");
        }
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        String tenantCode = authorizedUser.getTenantCode();
        Account result = accountService.findOneByUsernameV2(tenantCode, username);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity updateLastLoginTime(long accountId) {
        accountService.updateLastLoginTime(accountId);
        return ResponseEntity.ok();
    }

    @Override
    @ApiOperation(value = "检查当前账号是否确认过该网站的隐私协议")
    public ResponseEntity<Boolean> checkCurrentUserPrivacy(Long accountId, String webSite,
                                                           String version) {

        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        AccountPrivacyInfo request = new AccountPrivacyInfo();
        request.setAccountId(accountId);
        request.setWebSite(StringUtils.trimWhitespace(webSite));
        if (!StringUtils.isEmpty(version)) {
            request.setVersion(StringUtils.trimWhitespace(version));
        }
        Boolean checkResult = accountPrivacyService.checkAccountPrivacy(request);
        return ResponseEntity.ok(checkResult);
    }

    @Override
    @ApiOperation(value = "添加隐私协议")
    public ResponseEntity<String> addCurrentUserPrivacy(AccountModel.AccountPrivacyInfo request) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if (authorizedUser.getAccountId().equals(request.getAccountId())) {
            accountPrivacyService.addAccountPrivacy(request);
        } else {
            throw new IllegalArgumentException("账号信息与当前用户不匹配");
        }
        return ResponseEntity.ok();
    }

    @Override
    @ApiOperation(value = "查询帐号加密密码")
    public ResponseEntity<String> getEncryptPassword(Long accountId) {
        return ResponseEntity.ok(accountService.getEncryptedPwd(accountId));
    }

    @Override
    public ResponseEntity<String> updateDoubleAuth(long accountId,
        boolean status) {
        accountService.updateDoubleAuth(accountId, status);
        return ResponseEntity.ok();
    }

    @Override
    public ResponseEntity<String> updateBindAuth(long accountId,
        boolean status) {
        accountService.updateBindAuth(accountId, status);
        return ResponseEntity.ok();
    }
}
