package com.xforceplus.business.account.service;

import com.xforceplus.api.model.AccountPlatformRelModel.Request.ChangeState;
import com.xforceplus.api.model.AccountPlatformRelModel.Request.Query;
import com.xforceplus.api.model.AccountPlatformRelModel.Request.Save;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.dao.AccountDao;
import com.xforceplus.dao.AccountPlatformRelDao;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.AccountPlatformRel;
import com.xforceplus.entity.User;
import com.xforceplus.query.AccountPlatformRelQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;


/**
 * 帐号Service
 *
 * @author geewit
 */
@Service
public class AccountPlatformRelService {
    private final static Logger logger = LoggerFactory.getLogger(AccountPlatformRelService.class);

    private final AccountPlatformRelDao accountPlatformRelDao;
    @Resource
    private AccountDao accountDao;
    @Resource
    private UserService userService;

    public AccountPlatformRelService(AccountPlatformRelDao accountPlatformRelDao) {
        this.accountPlatformRelDao = accountPlatformRelDao;
    }


    @Transactional(rollbackFor = Exception.class)
    public AccountPlatformRel save(Save model) {
        Optional<Account> accountOptional = accountDao.findById(model.getAccountId());
        if (!accountOptional.isPresent()) {
            throw new IllegalArgumentException("传入accountId非法:" + model.getAccountId());
        }
        Long userId;
        try {
            userId = Long.parseLong(model.getUserId());
        } catch (NumberFormatException e) {
            userId = -1L;
        }
        Optional<User> user = userService.findUserById(userId);
        if (!user.isPresent()) {
            throw new IllegalArgumentException("传入userId非法:" + model.getUserId());
        }
        AccountPlatformRel accountPlatformRel = this.getAccountPlatformRel(model.getPlatformType(), model.getPlatformId());
        if (accountPlatformRel == null) {
            accountPlatformRel = new AccountPlatformRel();
            BeanUtils.copyProperties(model, accountPlatformRel);
            accountPlatformRel.setStatus(1);
            accountPlatformRel.setEnableTime(new Date());
            accountPlatformRel.setCreateTime(new Date());
            accountPlatformRel.setUpdateTime(new Date());
        } else {
            accountPlatformRel.setAccountId(model.getAccountId());
            accountPlatformRel.setUserId(model.getUserId());
            accountPlatformRel.setStatus(1);
            accountPlatformRel.setEnableTime(new Date());
            accountPlatformRel.setUpdateTime(new Date());
        }
        return accountPlatformRelDao.saveAndFlush(accountPlatformRel);
    }


    public AccountPlatformRel getInfoByPlatformId(int platformType, String platformId) {
        AccountPlatformRel accountPlatformRel = this.getAccountPlatformRel(platformType, platformId);
        if (accountPlatformRel == null) {
            throw new IllegalArgumentException("未找到实体");
        }
        return accountPlatformRel;
    }

    public AccountPlatformRel getAccountPlatformRel(int platformType, String platformId) {
        if (platformType <= 0 || StringUtils.isBlank(platformId)) {
            throw new IllegalArgumentException("参数不能为空或不正确(platformType)");
        }
        Optional<AccountPlatformRel> optional = accountPlatformRelDao.findOne(new Specification<AccountPlatformRel>() {
            @Override
            public Predicate toPredicate(Root<AccountPlatformRel> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.<Integer>get("platformType"), platformType));
                predicates.add(builder.equal(root.<String>get("platformId"), platformId));
                if (!predicates.isEmpty()) {
                    query.where(predicates.stream().toArray(Predicate[]::new));
                }
                return query.getRestriction();
            }
        });
        return optional.orElse(null);
    }


    public List<AccountPlatformRel> list(Long userId, Integer platformType) {
        List<AccountPlatformRel> list = accountPlatformRelDao.findAll(new Specification<AccountPlatformRel>() {
            @Override
            public Predicate toPredicate(Root<AccountPlatformRel> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.<Long>get("userId"), userId));
                predicates.add(builder.equal(root.<Integer>get("status"), 1));
                if (platformType != null) {
                    predicates.add(builder.equal(root.<Integer>get("platformType"), platformType));
                }
                if (!predicates.isEmpty()) {
                    query.where(predicates.stream().toArray(Predicate[]::new));
                }
                return query.getRestriction();
            }
        });
        return list;
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateState(ChangeState changeState) {
        Long userId;
        try {
            userId = Long.parseLong(changeState.getUserId());
        } catch (NumberFormatException e) {
            String message = e.getMessage() + ", id: " + changeState.getUserId();
            logger.warn(message);
            userId = null;
        }
        if (userId == null) {
            String message = "userId不能为空";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        Optional<User> optionalUser = userService.findUserById(userId);
        if (!optionalUser.isPresent()) {
            throw new IllegalArgumentException("userId有误!");
        }
        Query query = new Query();
        query.setUserId(changeState.getUserId());
        query.setPlatformType(changeState.getPlatformType());
        Specification<AccountPlatformRel> specification = AccountPlatformRelQueryHelper.querySpecification(query);
        List<AccountPlatformRel> list = accountPlatformRelDao.findAll(specification);
        if (!list.isEmpty()) {
            list.stream().forEach(rel -> {
                rel.setStatus(changeState.getStatus());
                rel.setUpdateTime(new Date());
                accountPlatformRelDao.saveAndFlush(rel);
            });
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public AccountPlatformRel saveAccountPlatformRel(AccountPlatformRel accountPlatformRel) {
        return accountPlatformRelDao.saveAndFlush(accountPlatformRel);
    }


    public Page<AccountPlatformRel> page(Query query, Pageable pageable) {
        Specification<AccountPlatformRel> specification = AccountPlatformRelQueryHelper.querySpecification(query);
        return accountPlatformRelDao.findAll(specification, pageable);
    }
}
