package com.xforceplus.business.account.service;

import com.xforceplus.api.model.AccountModel.AccountPrivacyInfo;
import com.xforceplus.dao.AccountDao;
import com.xforceplus.dao.AccountPrivacyDao;
import com.xforceplus.entity.AccountPrivacy;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.criteria.Predicate;

import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * @Author: HWT
 * @Date: 2021/8/27 14:37
 */

@Service
public class AccountPrivacyService {

    private final AccountPrivacyDao accountPrivacyDao;

    private final AccountDao accountDao;

    public AccountPrivacyService(AccountPrivacyDao accountPrivacyDao,
                                 AccountDao accountDao) {
        this.accountPrivacyDao = accountPrivacyDao;
        this.accountDao = accountDao;
    }

    public boolean checkAccountPrivacy(AccountPrivacyInfo queryInfo) {

        List<AccountPrivacy> accountPrivacies = this.findAllByQuery(queryInfo);
        return !CollectionUtils.isEmpty(accountPrivacies);

    }


    public List<AccountPrivacy> findAllByQuery(AccountPrivacyInfo queryInfo) {

        Specification<AccountPrivacy> specification = this.buildSpecificationByQuery(queryInfo);

        return this.accountPrivacyDao.findAll(specification);
    }

    /**
     * 支持重复操作，不做处理
     * @param privacyInfo
     */
    @Transactional(rollbackFor = Exception.class)
    public void addAccountPrivacy(AccountPrivacyInfo privacyInfo) {
        accountDao.findById(privacyInfo.getAccountId())
                .orElseThrow(() -> new IllegalArgumentException("未找到账号：" + privacyInfo.getAccountId()));

        List<AccountPrivacy> accountPrivacies = this.findAllByQuery(privacyInfo);
        if (CollectionUtils.isEmpty(accountPrivacies)) {
            //如果不存在，就新增
            AccountPrivacy accountPrivacy = new AccountPrivacy();
            accountPrivacy.setAccountId(privacyInfo.getAccountId());
            accountPrivacy.setWebSite(privacyInfo.getWebSite());
            accountPrivacy.setVersion(privacyInfo.getVersion());

            accountPrivacyDao.saveAndFlush(accountPrivacy);
        }
    }


    private Specification<AccountPrivacy> buildSpecificationByQuery(AccountPrivacyInfo queryInfo) {

        return (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();

            if (queryInfo.getAccountId() != null) {
                predicates.add(builder.equal(root.<String>get("accountId"), queryInfo.getAccountId()));
            }

            if (StringUtils.isNotEmpty(queryInfo.getWebSite())) {
                predicates.add(builder.equal(root.<String>get("webSite"), queryInfo.getWebSite()));
            }
            if (StringUtils.isNotEmpty(queryInfo.getVersion())) {
                predicates.add(builder.equal(root.<String>get("version"), queryInfo.getVersion()));
            }

            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.toArray(new Predicate[0]));
            }
            return criteriaQuery.getRestriction();
        };

    }

}
