package com.xforceplus.business.account.service;

import com.xforceplus.api.model.AccountModel;
import com.xforceplus.api.model.AccountModel.Request.*;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.account.dto.PasswordCheckResultDto;
import com.xforceplus.business.message.service.MessageService;
import com.xforceplus.business.messagebus.AccountPubSubService;
import com.xforceplus.dao.AccountDao;
import com.xforceplus.dao.user.AccountUserExtendDao;
import com.xforceplus.domain.account.AccountType;
import com.xforceplus.dto.user.AccountUserDTO;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.Tenant;
import com.xforceplus.feign.tenant.message.EmailContentFeignClient;
import com.xforceplus.feign.tenant.message.SmsMessageFeignClient;
import com.xforceplus.query.AccountQueryHelper;
import com.xforceplus.security.strategy.model.AccountLoginFailStrategy;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.utils.CryptUtils;
import com.xforceplus.utils.AESHelp;
import com.xforceplus.utils.PropertiesUtils;
import com.xforceplus.utils.RegExUtil;
import com.xforceplus.utils.password.MD5PwdUtil;
import com.xforececlound.message.model.EmailContentReq;
import com.xforececlound.message.model.SmsMessageReq;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;


/**
 * 帐号Service
 *
 * @author geewit
 */
@Service
public class AccountService {
    private final static Logger logger = LoggerFactory.getLogger(AccountService.class);

    private final AccountDao accountDao;

    private final EmailContentFeignClient emailContentFeignClient;

    private final SmsMessageFeignClient smsMessageFeignClient;

    private final MessageService messageService;

    private final RedisTemplate redisTemplate;

    private final AccountUserExtendDao accountUserExtendDao;

    private final AccountPubSubService accountPubSubService;

    private static final String WILMAR = "wilmar";

    @Value("${tenant.security.password.salt:}")
    private String salt;

    @Value("${tenant.center.user.create.old:true}")
    private boolean oldModelCreateUser;

    public AccountService(AccountDao accountDao, EmailContentFeignClient emailContentFeignClient,
                          SmsMessageFeignClient smsMessageFeignClient, MessageService messageService,
                          RedisTemplate redisTemplate,
                          AccountUserExtendDao accountUserExtendDao,
                          AccountPubSubService accountPubSubService) {
        this.accountDao = accountDao;
        this.emailContentFeignClient = emailContentFeignClient;
        this.smsMessageFeignClient = smsMessageFeignClient;
        this.messageService = messageService;
        this.redisTemplate = redisTemplate;
        this.accountUserExtendDao = accountUserExtendDao;
        this.accountPubSubService = accountPubSubService;
    }


    public Account findOneByLogin(Login login) {
        if (StringUtils.isBlank(login.getEmail()) && StringUtils.isBlank(login.getTelPhone()) && StringUtils.isBlank(login.getUsername())) {
            throw new IllegalArgumentException("用户名不能为空");
        }
        Account account = null;
        if (StringUtils.isNotBlank(login.getEmail())) {
            account = this.findAccountByUserName(login.getEmail());
        }
        if (account == null && StringUtils.isNotBlank(login.getTelPhone())) {
            account = this.findAccountByUserName(login.getTelPhone());
        }
        if (account == null && StringUtils.isNotBlank(login.getUsername())) {
            account = this.findAccountByUserName(login.getUsername());
        }
        if (account == null) {
            throw new IllegalArgumentException("无法找到账户");
        }
        return account;
    }


    public Account findOneByUsername(String username) {
        Specification<Account> specification = AccountQueryHelper.queryOneSpecification(null, username, false);
        List<Account> accounts = accountDao.findAll(specification);
        if (CollectionUtils.isEmpty(accounts)) {
            throw new IllegalArgumentException("未找到账户实体(" + username + ")");
        } else if (accounts.size() > 1) {
            logger.warn("用户名不唯一 username:{}", username);
        }
        return accounts.get(0);
    }

    public Account findOneByUsername(String tenantCode, String username) {
        Specification<Account> specification = AccountQueryHelper.queryOneSpecification(tenantCode, username, false);
        List<Account> accounts = accountDao.findAll(specification);
        if (CollectionUtils.isEmpty(accounts)) {
            throw new IllegalArgumentException("未找到账户实体(" + tenantCode + "," + username + ")");
        } else if (accounts.size() > 1) {
            logger.warn("用户名不唯一 tenantCode:{},username:{}", tenantCode, username);
        }
        return accounts.get(0);
    }

    public Account findOneByUsernameV2(String tenantCode, String username) {
        //TODO 1. 益海嘉里逻辑: 用户名上没有租户code！！！  如果是益海嘉里，检查的时候 要比对 tenantUserName, username
        if (WILMAR.equalsIgnoreCase(tenantCode) && StringUtils.isNotEmpty(username)) {
            return findOneByUsername(tenantCode, username);
        }
        //2. 手机号，邮箱，走原来逻辑
        if (RegExUtil.checkEmail(username) || RegExUtil.checkMobile(username)) {
            return findOneByUsername(tenantCode, username);
        }

        //3. 不是益海嘉里的用户名，用户名需要加上tenantCode,
        return findOneByUsername(tenantCode + username);
    }

    public Account findOneByQuery(Login query) {
        Specification<Account> specification = AccountQueryHelper.queryOneSpecification(query);

        List<Account> accountList = accountDao.findAll(specification);
        if (CollectionUtils.isEmpty(accountList)) {
            return null;
        } else if (accountList.size() > 1) {
            logger.warn("用户名不唯一：{}", query.toString());
        }
        return accountList.get(0);
    }

    public List<Account> findAllByQuery(Login query) {
        Specification<Account> specification = AccountQueryHelper.queryOneSpecification(query);
        List<Account> accountList = accountDao.findAll(specification);
        return accountList;
    }

    public List<Long> findTenantIdsByLogin(Login login) {
        login.setAttributes(Stream.of("tenantId").collect(Collectors.toSet()));
        return accountDao.findAttributes(login, Sort.unsorted());
    }

    @Transactional(rollbackFor = Exception.class)
    public void resetPassword(ChangePassword request) {
        String username = request.getUsername();
        Specification<Account> specification = AccountQueryHelper.queryOneSpecification(null, username, false);
        Account account = accountDao.findOne(specification).orElseThrow(() -> new IllegalArgumentException("未找到账号实体"));
        logger.info("username = " + username + ", salt = " + salt);
        String accountSalt = account.getSalt();
        boolean match = false;
        String rawPassword = this.safeDesPassword(request.getOriginalPassword());
        account.setRawPassword(rawPassword);
        String originalEncryptedPassword = CryptUtils.encryptPassword(accountSalt, rawPassword, salt);
        logger.debug("accountSalt = {}, originalEncryptedPassword = {}", accountSalt, originalEncryptedPassword);
        if (originalEncryptedPassword.equals(account.getPassword())) {
            match = true;
        }

        if (!match) {
            accountSalt = account.getUsername();
            originalEncryptedPassword = CryptUtils.encryptPassword(accountSalt, rawPassword, salt);
            logger.debug("accountSalt = {}, originalEncryptedPassword = {}", accountSalt, originalEncryptedPassword);
            if (originalEncryptedPassword.equals(account.getPassword())) {
                match = true;
                account.setSalt(account.getUsername());
            }
        }

        if (!match) {
            accountSalt = account.getEmail();
            originalEncryptedPassword = CryptUtils.encryptPassword(accountSalt, rawPassword, salt);
            logger.debug("accountSalt = {}, originalEncryptedPassword = {}", accountSalt, originalEncryptedPassword);
            if (originalEncryptedPassword.equals(account.getPassword())) {
                match = true;
                account.setSalt(account.getEmail());
            }
        }

        if (!match) {
            accountSalt = account.getTelPhone();
            originalEncryptedPassword = CryptUtils.encryptPassword(accountSalt, rawPassword, salt);
            logger.debug("accountSalt = {}, originalEncryptedPassword = {}", accountSalt, originalEncryptedPassword);
            if (originalEncryptedPassword.equals(account.getPassword())) {
                match = true;
                account.setSalt(account.getTelPhone());
            }
        }


        if (!match) {
            String oldEncryptedPassword = CryptUtils.encode(rawPassword);
            if (!oldEncryptedPassword.equals(account.getPassword())) {
                throw new IllegalArgumentException("未找到账号:" + request.getUsername());
            }
        }
        String encryptedPassword =
                CryptUtils.encryptPassword(account.getSalt(), safeDesPassword(request.getPassword()), salt);

        if (encryptedPassword.equals(account.getPassword())) {
            throw new IllegalArgumentException("新密码不能和原密码一样");
        }

        account.setPassword(encryptedPassword);
        account.setChangePasswordFlag(false);
        this.saveAndFlush(account);
    }


    /**
     * 华润逻辑
     *
     * @param request
     */
    @Transactional(rollbackFor = Exception.class)
    public void updatePasswordCiphertext(UpdatePasswordCiphertext request) {
        Account account = this.findById(request.getAccountId());
        if (request.getPassword().equals(account.getPassword())) {
            return;
        }
        logger.info("修改明文密码成功，accountId = {}" + request.getAccountId());
        account.setPassword(request.getPassword());
        account.setChangePasswordFlag(false);
        this.saveAndFlush(account);
    }

    @Transactional(rollbackFor = Exception.class)
    public void resetCurrentUserPassword(ChangeCurrentPassword request) {
        final String at = "@";
        //1. check user
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        //2. check captcha code
        boolean resp;
        if (request.getAccount().contains(at)) {
            resp = messageService.checkAuthEmailCode(request.getValidCode(), request.getAccount(), request.getMsgId());
        } else {
            resp = messageService.checkAuthSmsCode(request.getValidCode(), request.getAccount(), request.getMsgId());
        }
        if (!resp) {
            throw new IllegalArgumentException("验证码错误");
        }

        //3.check input pwd
        String decryptedPassword = this.desPassword(request.getPassword());


        Account account = accountDao.findById(authorizedUser.getAccountId()).orElseThrow(() -> new IllegalArgumentException("未找到账号实体(" + authorizedUser.getAccountId() + ")"));

        String dynamicSalt = account.getSalt();
        if (StringUtils.isBlank(dynamicSalt)) {
            if (StringUtils.isNotBlank(account.getUsername())) {
                dynamicSalt = account.getUsername();
            } else if (StringUtils.isNotBlank(account.getTelPhone())) {
                dynamicSalt = account.getTelPhone();
            } else if (StringUtils.isNotBlank(account.getEmail())) {
                dynamicSalt = account.getEmail();
            }
        }

        //4. update pwd
        String encryptedPassword = CryptUtils.encryptPassword(dynamicSalt, decryptedPassword, this.salt);
        //密码一样就跳过
        if (encryptedPassword.equals(account.getPassword())) {
            throw new IllegalArgumentException("新密码不能和原密码一样");
        }
        account.setRawPassword(decryptedPassword);
        account.setPassword(encryptedPassword);
        account.setSalt(dynamicSalt);
        this.saveAndFlush(account);
    }


    public Page<Account> page(Query query, Pageable pageable) {
        Specification<Account> specification = AccountQueryHelper.querySpecification(query);
        return accountDao.findAll(specification, pageable);
    }

    public Page<Account> page(Specification<Account> specification, Pageable pageable) {
        return accountDao.findAll(specification, pageable);
    }

    public List<Account> list(Query query, Sort sort) {
        Specification<Account> specification = AccountQueryHelper.querySpecification(query);
        Iterable<Account> iterable = accountDao.findAll(specification, sort);
        List<Account> list = StreamSupport.stream(iterable.spliterator(), true).collect(Collectors.toList());
        return list;
    }

    public List<Account> list(Specification<Account> specification, Sort sort) {
        return accountDao.findAll(specification, sort);
    }

    @Transactional(rollbackFor = Exception.class)
    public Account create(Create model) {
        return this.save(null, model, model.isEnableSendMsg(), model.isRandomPassword(), false, model.getUpdateIgnoreProperties());
    }

    @Transactional(rollbackFor = Exception.class)
    public <S extends Save> Account saveOriginPassword(S model) {
        Account entity = null;
        long accountId = 0;
        if (model instanceof Update) {
            accountId = ((Update) model).getAccountId();
        }
        if (accountId > 0) {
            entity = this.findById(accountId);
        } else {
            if (StringUtils.isNotBlank(model.getTelPhone())) {
                entity = accountDao.findByTelPhone(model.getTelPhone());
            }
            if (entity == null && StringUtils.isNotBlank(model.getEmail())) {
                entity = accountDao.findByEmail(model.getEmail());
            }
            if (entity == null) {
                entity = new Account();
            }
        }
        BeanUtils.copyProperties(model, entity);
        return accountDao.saveAndFlush(entity);
    }

    public boolean validPassword(AccountModel.Request.Login login, Account existAccount) {
        PasswordCheckResultDto passwordCheckResultDto = this.checkPassword(login.getPassword(), existAccount);
        if (passwordCheckResultDto.isPwdCorrect()) {
            return true;
        } else {
            String message = "用户名密码错误";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }
    }


    /**
     * @param model
     * @param isMergeAccount 是否覆盖修改
     * @param <S>
     * @return
     */
    public <S extends Save> Account save(Tenant tenant, S model, boolean isEnableSendMsg, boolean isRandomPassword, boolean isMergeAccount, List<String> updateIgnoreProperties) {
        logger.info("isEnableSendMsg = {}, isRandomPassword = {}, isMergeAccount = {}", isEnableSendMsg, isRandomPassword, isMergeAccount);
        Account entity;
        long accountId = 0;
        model.fillSalt();
        if (model instanceof Update) {
            accountId = ((Update) model).getAccountId();
        }

        String username = model.getUsername();
        boolean updateTelphone = true;
        boolean updateEmail = true;
        boolean updateUsername = true;
        boolean updatePassword = true;
        boolean updateSalt = true;
        boolean isNew;

        String tenantCode = tenant != null ? StringUtils.defaultString(tenant.getTenantCode()) : StringUtils.EMPTY;
        if (AccountType.OTHER.equals(model.getType()) && !WILMAR.equalsIgnoreCase(tenantCode) && StringUtils.isNotEmpty(model.getUsername())) {
            if (model.getUsername() != null && !model.getUsername().startsWith(tenantCode)) {
                username = tenantCode + model.getUsername();
            }
        }

        if (accountId > 0) {
            entity = this.findById(accountId);
            isNew = false;
        } else {
            Login query = new Login();
            query.setEmail(model.getEmail());
            query.setTelPhone(model.getTelPhone());
            query.setUsername(username);
            entity = this.findOneByQuery(query);
            if (entity == null) {
                entity = new Account();
                isNew = true;
            } else if (!isMergeAccount) {
                String message = "已存在该账号(email:" + model.getEmail() + ",telPhone:" + model.getTelPhone() + ",username:" + model.getUsername() + ")";
                throw new IllegalArgumentException(message);
            } else {
                isNew = false;
            }
        }
        if (tenant != null) {
            entity.setTenantId(tenant.getTenantId());
        }
        if (StringUtils.isBlank(model.getEmail())) {
            model.setEmail(null);
        }
        if (StringUtils.isBlank(model.getTelPhone())) {
            model.setTelPhone(null);
        }
        if (StringUtils.isBlank(model.getUsername())) {
            model.setUsername(null);
        }
        if (isNew) {
            BeanUtils.copyProperties(model, entity, Stream.of("username").toArray(String[]::new));
        } else {
            if (!CollectionUtils.isEmpty(updateIgnoreProperties)) {
                updateIgnoreProperties.add("username");
            } else {
                updateIgnoreProperties = Stream.of("username").collect(Collectors.toList());
            }
            BeanUtils.copyProperties(model, entity, updateIgnoreProperties.stream().distinct().toArray(String[]::new));

            if (!CollectionUtils.isEmpty(updateIgnoreProperties)) {
                for (String updateIgnoreProperty : updateIgnoreProperties) {
                    if ("telphone".equalsIgnoreCase(updateIgnoreProperty)) {
                        updateTelphone = false;
                    } else if ("email".equalsIgnoreCase(updateIgnoreProperty)) {
                        updateEmail = false;
                    } else if ("username".equalsIgnoreCase(updateIgnoreProperty)) {
                        updateUsername = false;
                    } else if ("password".equalsIgnoreCase(updateIgnoreProperty)) {
                        updatePassword = false;
                    } else if ("salt".equalsIgnoreCase(updateIgnoreProperty)) {
                        updateSalt = false;
                    }
                }
            }
        }

        if (AccountType.OTHER.equals(model.getType())) {
            if (updateUsername) {
                entity.setUsername(username);
            }
        } else {
            if (StringUtils.isNotBlank(model.getTelPhone())) {
                if (updateTelphone) {
                    entity.setTelPhone(model.getTelPhone());
                }
            }
            if (StringUtils.isNotBlank(model.getEmail())) {
                if (updateEmail) {
                    entity.setEmail(model.getEmail());
                }

            }
            if (StringUtils.isBlank(model.getTelPhone()) && StringUtils.isBlank(model.getEmail())) {
                if (StringUtils.isNotBlank(model.getUsername())) {
                    if (StringUtils.contains(model.getUsername(), Separator.AT)) {
                        if (StringUtils.isBlank(entity.getEmail()) && updateEmail) {
                            entity.setEmail(model.getUsername());
                        }
                    } else if (RegExUtil.checkMobile(model.getUsername())) {
                        if (StringUtils.isBlank(entity.getTelPhone()) && updateTelphone) {
                            entity.setTelPhone(model.getUsername());
                        }
                    }
                    if (username == null) {
                        username = model.getUsername();
                    }
                } else {
                    throw new IllegalArgumentException("没有定义合法用户名");
                }
            }
        }
        Long tenantId = null;
        String tenantName = null;
        if (tenant != null) {
            tenantId = tenant.getTenantId();
            tenantName = tenant.getTenantName();
            tenantCode = tenant.getTenantCode();
        } else {
            IAuthorizedUser authorizedUser = UserInfoHolder.get();
            if (authorizedUser != null) {
                tenantId = authorizedUser.getTenantId();
                tenantName = authorizedUser.getTenantName();
                tenantCode = authorizedUser.getTenantCode();
            }
        }
        if (model instanceof Create) {
            String password;
            if (StringUtils.isBlank(((Create) model).getPassword())) {
                if (isRandomPassword || ((Create) model).isRandomPassword()) {
                    password = RandomStringUtils.randomAlphabetic(8);
                    logger.info("用户(username:{}, telphone:{}, email:{})的随机密码:{}", model.getUsername(), model.getTelPhone(), model.getEmail(), password);
                } else {
                    throw new IllegalArgumentException("没有定义密码");
                }
            } else {
                password = ((Create) model).getPassword();
                entity.setRawPassword(password);
            }
            if (model.getSalt() != null && updatePassword) {
                String encryptedPassword = CryptUtils.encryptPassword(model.getSalt(), password, salt);
                entity.setPassword(encryptedPassword);
            }
            if (updateSalt) {
                entity.setSalt(model.getSalt());
            }
            if (isEnableSendMsg) {
                try {
                    logger.info("isEnableSendMsg, this.sendMessage(tenantId:{}, email:{}, telphone:{}, password:*, tenantName:{}, username:{})", tenantId, entity.getEmail(), entity.getTelPhone(), tenantName, username);
                    this.sendMessage(tenantId, entity.getEmail(), entity.getTelPhone(), password, tenantName, username);
                } catch (Exception e) {
                    logger.warn("信息发送异常", e);
                }

            }
        }

        if (tenantId != null) {
            if (isNew) {
                if (WILMAR.equalsIgnoreCase(tenantCode) && oldModelCreateUser) {
                    entity.setUsername(model.getUsername());
                }
            }
            entity.setTenantId(tenantId);
            entity.setDoubleAuthFlag(false);
        }

        //fix bug: username not set
        if (AccountType.OTHER.equals(model.getType())) {
            if (model.getUsername() != null && updateUsername) {
                entity.setUsername(username);
            }
        } else {
            //是phone_email 的账号，只有用户名 不是手机号，邮箱，才设置用户名
            if (model.getUsername() != null) {
                boolean isNotPhone = !model.getUsername().equalsIgnoreCase(entity.getTelPhone());
                boolean isNotEmail = !model.getUsername().equalsIgnoreCase(entity.getEmail());
                if (isNotPhone && isNotEmail) {
                    entity.setUsername(username);
                }
            }
        }

        return this.saveAndFlush(entity);
    }

    @Transactional(rollbackFor = Exception.class)
    public Account update(long accountId, Update model) {
        Account existEntity = this.findById(accountId);
        BeanUtils.copyProperties(model, existEntity, Stream.of("accountId", "password", "salt").toArray(String[]::new));

        //Fix bug: 判斷 用戶名，手機號，郵箱是否重複
        Login query = new Login();
        query.setEmail(model.getEmail());
        query.setTelPhone(model.getTelPhone());
        query.setUsername(model.getUsername());
        List<Account> accountList = this.findAllByQuery(query);
        if (accountList != null) {
            for (int i = 0; i < accountList.size(); i++) {
                Account account = accountList.get(i);
                //查詢到的賬號，不是當前需要更新的賬號
                if (!account.getAccountId().equals(existEntity.getAccountId())) {
                    throw new IllegalArgumentException(" 账号已经存在: (" + model.getEmail() + "," + model.getTelPhone() + "," + model.getUsername() + ")");
                }
            }
        }


        return this.saveAndFlush(existEntity);
    }

    /**
     * 如果数据库中已经存在 telphone或email就直接返回, 不存在就保存
     *
     * @param telphone
     * @param email
     * @return
     */
    public Account create(Tenant tenant, String telphone, String email, String username, String password, Integer status, AccountType type, boolean isEnableSendMsg, boolean isRandomPassword, boolean isMergeAccount, boolean changePasswordFlag, List<String> updateIgnoreProperties) {
        logger.info("telphone = {}, email = {}, username = {}, status = {}, isMergeAccount = {}, isEnableSendMsg = {}, changePasswordFlag = {}", telphone, email, username, status, isMergeAccount, isEnableSendMsg, changePasswordFlag);
        if (StringUtils.isBlank(telphone) && StringUtils.isBlank(email) && StringUtils.isBlank(username)) {
            return null;
        }
        Login query = new Login();
        if (StringUtils.isNotBlank(telphone)) {
            query.setTelPhone(telphone);
        }
        if (StringUtils.isNotBlank(email)) {
            query.setEmail(email);
        }
        if (StringUtils.isNotBlank(username)) {
            query.setUsername(username);
            if (!WILMAR.equalsIgnoreCase(tenant.getTenantCode()) && !username.startsWith(tenant.getTenantCode())) {
                if (!RegExUtil.checkEmail(username) && !RegExUtil.checkMobile(username)) {
                    query.setUsername(tenant.getTenantCode() + username);
                }
            }
        }
        List<Account> accountList = this.findAllByQuery(query);
        Account account = null;

        if (CollectionUtils.isEmpty(accountList)) {
            Create newAccount = new Create();
            newAccount.setTelPhone(telphone);
            newAccount.setEmail(email);
            newAccount.setUsername(username);
            newAccount.setPassword(password);
            newAccount.setStatus(status);
            newAccount.setType(type);
            newAccount.setEnableSendMsg(isEnableSendMsg);
            newAccount.setChangePasswordFlag(changePasswordFlag);
            account = this.save(tenant, newAccount, isEnableSendMsg, isRandomPassword, isMergeAccount, updateIgnoreProperties);
        } else {
            String message = "账号已经存在: (" + email + "," + telphone + "," + username + ")";
            logger.warn(message);
            if (accountList.size() > 1) {
                throw new IllegalArgumentException(message);
            }
            if (!isMergeAccount) {
                throw new IllegalArgumentException(message);
            }
            account = accountList.get(0);
            if (RegExUtil.checkEmail(username)) {
                email = username;
            } else if (RegExUtil.checkMobile(username)) {
                telphone = username;
            } else if (tenant != null && StringUtils.isNotBlank(username)) {
                if (username.contains(tenant.getTenantCode())) {
                    account.setUsername(username);
                } else {
                    if (WILMAR.equalsIgnoreCase(tenant.getTenantCode())) {
                        account.setUsername(username);
                    } else {
                        account.setUsername(tenant.getTenantCode() + username);
                    }
                }
            }
            if (StringUtils.isNotEmpty(telphone)) {
                account.setTelPhone(telphone);
            }
            if (StringUtils.isNotEmpty(email)) {
                account.setEmail(email);
            }

            account = this.saveAndFlush(account);
        }
        return account;
    }

    public Account findById(long accountId) {
        return accountDao.findById(accountId).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long accountId) {
        Account account = accountDao.findById(accountId).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
        accountDao.deleteById(accountId);
    }

    @Transactional(rollbackFor = Exception.class)
    public Account updateStatus(long accountId, int status) {
        IAuthorizedUser currentUser = UserInfoHolder.get();
        if (currentUser != null && accountId == currentUser.getAccountId().longValue()) {
            throw new IllegalArgumentException("操作失败！不能锁定/禁用自己的账号!");
        }
        Account account = accountDao.findById(accountId).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
        account.setStatus(status);
        // 启用/解锁 时需要清除redis key
        if (1 == status) {
            try {
                String username = AccountLoginFailStrategy.ACCOUNT_LOGIN_FAILS_PREFIX + account.getUsername();
                String mobile = AccountLoginFailStrategy.ACCOUNT_LOGIN_FAILS_PREFIX + account.getTelPhone();
                String email = AccountLoginFailStrategy.ACCOUNT_LOGIN_FAILS_PREFIX + account.getEmail();
                String[] keys = {username, mobile, email};
                redisTemplate.delete(Arrays.asList(keys));
            } catch (Exception e) {
                logger.warn(e.getMessage());
            }
        }
        return this.saveAndFlush(account);
    }

    public void sendMessage(Long tenantId, String email, String telPhone, String plantPassword, String tenantName, String username) {
        logger.info("email = {}, telPhone = {}, username = {}", email, telPhone, username);
        if (StringUtils.isBlank(email) && StringUtils.isBlank(telPhone) && StringUtils.isBlank(username)) {
            logger.info("StringUtils.isBlank(email) && StringUtils.isBlank(telPhone) && StringUtils.isBlank(username), return");
            return;
        }
        if (tenantId == null) {
            tenantId = 0L;
        }
        //并发送邮件
        if (RegExUtil.checkEmail(email)) {
            Map<String, String> emailParam = new HashMap<>();
            emailParam.put("account", email);
            emailParam.put("password", plantPassword);
            emailParam.put("tenantName", tenantName);
            emailParam.put("userName", username);
            EmailContentReq emailContentReq = new EmailContentReq();
            emailContentReq.setEmail(email);
            emailContentReq.setTemplateCode(PropertiesUtils.EMAIL_USER_CREATE);
            emailContentReq.setTenantId(tenantId);
            emailContentReq.setProps(emailParam);
            emailContentReq.setAppId(1L);
            logger.info("send message to email = {}", email);
            emailContentFeignClient.send(String.valueOf(tenantId), emailContentReq);
        } else if (RegExUtil.checkMobile(telPhone)) {
            //发送短信
            Map<String, String> smsParams = new HashMap<>();
            smsParams.put("password", plantPassword);
            smsParams.put("tenant", tenantName);
            smsParams.put("username", telPhone);
            SmsMessageReq smsMessageReq = new SmsMessageReq();
            smsMessageReq.setMobile(telPhone);
            smsMessageReq.setTemplateCode(PropertiesUtils.SMS_USER_CREATE);
            smsMessageReq.setTemplateParamJson(JsonUtils.toJson(smsParams));
            smsMessageReq.setTenantId(tenantId);
            smsMessageReq.setSignName("票税助手");
            logger.info("send message to telphone = {}", telPhone);
            smsMessageFeignClient.send(String.valueOf(tenantId), smsMessageReq);
        } else {
            logger.warn("do not send any message, because of none email or telphone");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public Account checkPasswordByAccountId(Long accountId, String password) {

        Optional<Account> optionalAccount = accountDao.findById(accountId);
        if (!optionalAccount.isPresent()) {
            throw new IllegalArgumentException("不存在的用户");
        }
        Account account = optionalAccount.get();
        PasswordCheckResultDto passwordCheckResultDto = checkPassword(password, account);

        if (passwordCheckResultDto.isPwdCorrect()) {
            account = passwordCheckResultDto.getAccount();
            return account;
        } else {
            throw new IllegalArgumentException("密码错误");
        }
    }

    private Account findAccountByUserName(String username) {

        List<Account> accountList = accountDao.findAllByLoginName(username);
        if (accountList == null || accountList.isEmpty()) {
            return null;
        } else if (accountList.size() > 1) {
            throw new IllegalArgumentException("用户数据错误，请联系管理员核对账号信息！");
        }
        return accountList.get(0);
    }

    private PasswordCheckResultDto checkPassword(String inputPassword, Account existAccount) {
        PasswordCheckResultDto passwordCheckResultDto = new PasswordCheckResultDto();
        passwordCheckResultDto.setAccount(existAccount);

        //解密前端密码
        String decryptedPassword;
        try {
            decryptedPassword = this.desPassword(inputPassword);
        } catch (Exception e) {
            decryptedPassword = inputPassword;
        }

        //entity salt有值
        if (StringUtils.isNotBlank(existAccount.getSalt())) {
            String encryptPassword = CryptUtils.encryptPassword(existAccount.getSalt(), decryptedPassword, this.salt);
            if (encryptPassword.equals(existAccount.getPassword())) {
                passwordCheckResultDto.setPwdCorrect(true);
                return passwordCheckResultDto;
            }
        }
        //2. 盐值为空，也需要判断是否有盐值加密的
        if (StringUtils.isNotBlank(existAccount.getEmail()) && !StringUtils.equals(existAccount.getEmail(), existAccount.getSalt())) {
            String encodePwd = CryptUtils
                    .encryptPassword(existAccount.getEmail(), decryptedPassword, this.salt);
            if (encodePwd.equals(existAccount.getPassword())) {
                passwordCheckResultDto.setPwdCorrect(true);
                existAccount.setPassword(encodePwd);
                existAccount.setSalt(existAccount.getEmail());
                return passwordCheckResultDto;
            }
        }
        if (StringUtils.isNotBlank(existAccount.getTelPhone()) && !StringUtils.equals(existAccount.getTelPhone(), existAccount.getSalt())) {
            String encodePwd = CryptUtils
                    .encryptPassword(existAccount.getTelPhone(), decryptedPassword, this.salt);
            if (encodePwd.equals(existAccount.getPassword())) {
                passwordCheckResultDto.setPwdCorrect(true);
                existAccount.setPassword(encodePwd);
                existAccount.setSalt(existAccount.getTelPhone());
                return passwordCheckResultDto;
            }
        }
        if (StringUtils.isNotBlank(existAccount.getUsername()) && !StringUtils.equals(existAccount.getUsername(), existAccount.getSalt())) {
            String encodePwd = CryptUtils
                    .encryptPassword(existAccount.getUsername(), decryptedPassword, this.salt);
            if (encodePwd.equals(existAccount.getPassword())) {
                passwordCheckResultDto.setPwdCorrect(true);
                existAccount.setPassword(encodePwd);
                existAccount.setSalt(existAccount.getUsername());
                return passwordCheckResultDto;
            }
        }

        //没有salt，兼容老的加密方式
        try {
            String md5SecretPassWord = MD5PwdUtil.encode(decryptedPassword);
            this.loginResetPassword(passwordCheckResultDto, existAccount, decryptedPassword);
            if (passwordCheckResultDto.isPwdCorrect()) {
                return passwordCheckResultDto;
            }
        } catch (Exception e) {
            logger.warn("md5加密异常");
        }


        passwordCheckResultDto.setPwdCorrect(false);
        return passwordCheckResultDto;
    }

    /**
     * 登录并且重置密码
     *
     * @param passwordCheckResultDto
     * @param existAccount
     * @param decryptedPassword
     * @return
     */
    private PasswordCheckResultDto loginResetPassword(PasswordCheckResultDto passwordCheckResultDto, Account existAccount, String decryptedPassword) {
        String usesername = existAccount.getUsername();
        if (StringUtils.isBlank(usesername)) {
            if (StringUtils.isNotBlank(existAccount.getTelPhone())) {
                usesername = existAccount.getTelPhone();
            } else {
                usesername = existAccount.getEmail();
            }
        }

        if (StringUtils.equals(existAccount.getPassword(), MD5PwdUtil.encode(decryptedPassword))) {
            String encryptedPassword = CryptUtils.encryptPassword(usesername, decryptedPassword, salt);
            existAccount.setSalt(usesername);
            existAccount.setPassword(encryptedPassword);
            passwordCheckResultDto.setPwdCorrect(true);
        } else {
            passwordCheckResultDto.setPwdCorrect(false);
        }
        return passwordCheckResultDto;
    }


    @Transactional(rollbackFor = Exception.class)
    public Account checkPasswordByUserName(String username, String password) {

        Account account = this.findAccountByUserName(username);
        if (account == null) {
            throw new IllegalArgumentException("不存在的用户");
        }

        PasswordCheckResultDto passwordCheckResultDto = this.checkPassword(password, account);

        if (passwordCheckResultDto.isPwdCorrect()) {
            account = passwordCheckResultDto.getAccount();
            return account;
        } else {
            throw new IllegalArgumentException("密码错误");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void changePhone(ChangePhone request) {
        if (!RegExUtil.checkMobile(request.getPhone())) {
            throw new IllegalArgumentException("手机号码格式不正确");
        }
        if (logger.isDebugEnabled()) {
            logger.info("上下文信息：{}", UserInfoHolder.currentUser());
        }
        Long accountId = UserInfoHolder.currentUser().getAccountId();

        boolean resp = messageService.checkAuthSmsCode(request.getCaptcha(), request.getPhone(), request.getMsgId());
        if (!resp) {
            throw new IllegalArgumentException("验证码错误");
        }
        String password;
        try {
            password = this.desPassword(request.getPassword());
        } catch (Exception e) {
            password = request.getPassword();
        }
        Account account = this.checkPasswordByAccountId(accountId, request.getPassword());

        String encryptPassword = CryptUtils.encryptPassword(request.getPhone(), password, this.salt);
        account.setPassword(encryptPassword);
        account.setSalt(request.getPhone());
        account.setTelPhone(request.getPhone());
        account.setBindAuthFlag(Boolean.FALSE);
        this.saveAndFlush(account);

    }

    @Transactional(rollbackFor = Exception.class)
    public void bindPhone(BindPhone request) {
        if (!RegExUtil.checkMobile(request.getPhone())) {
            throw new IllegalArgumentException("手机号码格式不正确");
        }

        boolean resp = messageService.checkAuthSmsCode(request.getCaptcha(), request.getPhone(), request.getMsgId());
        if (!resp) {
            throw new IllegalArgumentException("验证码错误");
        }

        Account account = this.checkPasswordByUserName(request.getUsername(), request.getPassword());

        String password = request.getPassword();
        try {
            password = this.desPassword(request.getPassword());
        } catch (Exception e) {

        }
        String encryptPassword = CryptUtils.encryptPassword(request.getPhone(), password, this.salt);
        account.setPassword(encryptPassword);
        account.setTelPhone(request.getPhone());
        account.setSalt(request.getPhone());
        account.setBindAuthFlag(Boolean.FALSE);
        this.saveAndFlush(account);

    }

    @Transactional(rollbackFor = Exception.class)
    public void changeEmail(ChangeEmail request) {

        if (!RegExUtil.checkEmail(request.getEmail())) {
            throw new IllegalArgumentException("邮箱格式不正确");
        }

        Long accountId = UserInfoHolder.currentUser().getAccountId();

        boolean resp = messageService.checkAuthEmailCode(request.getCaptcha(), request.getEmail(), request.getMsgId());
        if (!resp) {
            throw new IllegalArgumentException("验证码错误");
        }

        String password = request.getPassword();
        try {
            password = this.desPassword(request.getPassword());
        } catch (Exception e) {

        }
        Account account = this.checkPasswordByAccountId(accountId, request.getPassword());

        String encryptPassword = CryptUtils.encryptPassword(request.getEmail(), password, this.salt);
        account.setPassword(encryptPassword);
        account.setEmail(request.getEmail());
        account.setSalt(account.getEmail());
        account.setBindAuthFlag(Boolean.FALSE);
        this.saveAndFlush(account);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindEmail(BindEmail request) {

        if (!RegExUtil.checkEmail(request.getEmail())) {
            throw new IllegalArgumentException("邮箱格式不正确");
        }

        boolean resp = messageService.checkAuthEmailCode(request.getCaptcha(), request.getEmail(), request.getMsgId());
        if (!resp) {
            throw new IllegalArgumentException("验证码错误");
        }

        String password = request.getPassword();
        try {
            password = this.desPassword(request.getPassword());
        } catch (Exception e) {

        }
        Account account = this.checkPasswordByUserName(request.getUsername(), request.getPassword());

        String encryptPassword = CryptUtils.encryptPassword(request.getEmail(), password, this.salt);
        account.setPassword(encryptPassword);
        account.setEmail(request.getEmail());
        account.setSalt(account.getEmail());
        account.setBindAuthFlag(Boolean.FALSE);
        this.saveAndFlush(account);
    }


    private String desPassword(String password) {
        try {
            return AESHelp.desEncrypt(AESHelp.CREDENTIAL_KEY, password);
        } catch (Exception e) {
            logger.debug("密码解析错误:{}", e.getMessage());
            throw new IllegalArgumentException("密码解析错误");
        }
    }

    private String safeDesPassword(String password) {
        try {
            return AESHelp.desEncrypt(AESHelp.CREDENTIAL_KEY, password);
        } catch (Exception e) {
            return password;
        }
    }

    public String sendChangePhoneCode(String phone) {
        if (!RegExUtil.checkMobile(phone)) {
            throw new IllegalArgumentException("手机号码格式不正确");
        }
        Account account = accountDao.findByTelPhone(phone);
        if (null != account) {
            throw new IllegalArgumentException("该手机号已存在");
        }

        return messageService.sendAuthSmsCode(phone);
    }

    public String sendChangeEmailCode(String email) {
        if (!RegExUtil.checkEmail(email)) {
            throw new IllegalArgumentException("邮箱格式不正确");
        }
        Account account = accountDao.findByEmail(email);
        if (null != account) {
            throw new IllegalArgumentException("该邮箱已存在");
        }

        return messageService.sendAuthEmailCode(email);
    }

    public String sendValidCode(String username) {

        Account account = this.findAccountByUserName(username);

        if (account == null) {
            throw new IllegalArgumentException("不存在的用户");
        }
        boolean isEmail = false;
        boolean isPhone = false;
        String email = null;
        String phone = null;

        //1. 优先使用 输入的邮箱或者手机号
        if (RegExUtil.checkEmail(username)) {
            isEmail = true;
            email = username;
        } else if (RegExUtil.checkMobile(username)) {
            isPhone = true;
            phone = username;
        } else {
            //2. 使用数据库内容
            if (account.getTelPhone() != null && RegExUtil.checkMobile(account.getTelPhone())) {
                isPhone = true;
                phone = account.getTelPhone();
            } else if (account.getEmail() != null && RegExUtil.checkEmail(account.getEmail())) {
                isEmail = true;
                email = account.getEmail();
            }
        }

        if (isEmail) {
            return messageService.sendAuthEmailCode(email);
        } else if (isPhone) {
            return messageService.sendAuthSmsCode(phone);
        } else {
            throw new IllegalArgumentException("不存在的账号");
        }

    }

    @Transactional(rollbackFor = Exception.class)
    public void updatePwd(AccountModel.ChangePwd request) {

        Account account = this.findAccountByUserName(request.getUsername());

        if (account == null) {
            throw new IllegalArgumentException("不存在的用户");
        }

        boolean isEmail = false;
        boolean isPhone = false;
        String email = null;
        String phone = null;
        //1. 优先使用 输入的邮箱或者手机号
        if (RegExUtil.checkEmail(request.getUsername())) {
            isEmail = true;
            email = request.getUsername();
        } else if (RegExUtil.checkMobile(request.getUsername())) {
            isPhone = true;
            phone = request.getUsername();
        } else {
            //2. 使用数据库内容
            if (account.getTelPhone() != null && RegExUtil.checkMobile(account.getTelPhone())) {
                isPhone = true;
                phone = account.getTelPhone();
            } else if (account.getEmail() != null && RegExUtil.checkEmail(account.getEmail())) {
                isEmail = true;
                email = account.getEmail();
            }
        }

        boolean resp;
        if (isEmail) {
            resp = messageService.checkAuthEmailCode(request.getValidCode(), email, request.getMsgId());
        } else if (isPhone) {
            resp = messageService.checkAuthSmsCode(request.getValidCode(), phone, request.getMsgId());
        } else {
            throw new IllegalArgumentException("手机号/邮箱格式不正确");
        }

        if (!resp) {
            throw new IllegalArgumentException("验证码错误");
        }

        String rawPassword = this.safeDesPassword(request.getPassword());

        String password = CryptUtils.encryptPassword(request.getUsername(), rawPassword, this.salt);
        //密码一样跳过
        if (password.equals(account.getPassword())) {
            return;
        }
        account.setRawPassword(rawPassword);
        account.setPassword(password);

        if (isEmail) {
            account.setEmail(email);
        } else if (isPhone) {
            account.setTelPhone(phone);
        } else {
            account.setUsername(request.getUsername());
        }

        account.setSalt(request.getUsername());
        this.saveAndFlush(account);
    }

    public List<Account> findByTenantCodeAndUsername(String tenantCode, String username) {
        Specification<Account> specification = AccountQueryHelper.queryOneSpecification(tenantCode, username, false);
        List<Account> accounts = accountDao.findAll(specification);
        if (accounts == null || accounts.isEmpty()) {
            throw new IllegalArgumentException("未找到账户实体(tenantCode:" + tenantCode + ",username:" + username);
        }
        return accounts;
    }

    @Transactional(rollbackFor = Exception.class)
    public void unBindPhone(UnBindPhone request) {

        Long accountId = UserInfoHolder.currentUser().getAccountId();

        Account account = this.findById(accountId);

        String password = request.getPassword();
        try {
            password = this.desPassword(request.getPassword());
        } catch (Exception e) {

        }
        String validPassword = CryptUtils.encryptPassword(account.getSalt(), password, this.salt);

        if (StringUtils.isBlank(account.getTelPhone())) {
            throw new IllegalArgumentException("未绑定手机号，不能解绑");
        }

        if (!account.getPassword().equalsIgnoreCase(validPassword)) {
            throw new IllegalArgumentException("密码错误");
        }

        if (StringUtils.isBlank(account.getUsername()) && StringUtils.isBlank(account.getEmail())) {
            throw new IllegalArgumentException("没有域账号与邮箱，手机号不能解绑");
        }

        String accountSalt = account.getSalt();
        if (account.getTelPhone().equals(account.getSalt())) {
            accountSalt = StringUtils.isNotBlank(account.getUsername()) ? account.getUsername() : account.getEmail();
        }

        String encryptPassword = CryptUtils.encryptPassword(accountSalt, password, this.salt);
        account.setPassword(encryptPassword);
        account.setTelPhone("");
        account.setSalt(accountSalt);
        this.saveAndFlush(account);
    }

    @Transactional(rollbackFor = Exception.class)
    public void unBindEmail(UnBindEmail request) {

        Long accountId = UserInfoHolder.currentUser().getAccountId();

        Account account = this.findById(accountId);

        String password = request.getPassword();
        try {
            password = this.desPassword(request.getPassword());
        } catch (Exception ignored) {
        }

        String validPassword = CryptUtils.encryptPassword(account.getSalt(), password, this.salt);

        if (!account.getPassword().equalsIgnoreCase(validPassword)) {
            throw new IllegalArgumentException("密码错误");
        }

        if (StringUtils.isBlank(account.getEmail())) {
            throw new IllegalArgumentException("未绑定邮箱，不能解绑");
        }

        if (StringUtils.isBlank(account.getUsername()) && StringUtils.isBlank(account.getTelPhone())) {
            throw new IllegalArgumentException("没有域账号与邮箱，手机号不能解绑");
        }

        String accountSalt = account.getSalt();
        if (account.getEmail().equals(account.getSalt())) {
            accountSalt = StringUtils.isNotBlank(account.getUsername()) ? account.getUsername() : account.getTelPhone();
        }

        String encryptPassword = CryptUtils.encryptPassword(accountSalt, password, this.salt);
        account.setPassword(encryptPassword);
        account.setEmail("");
        account.setSalt(accountSalt);
        this.saveAndFlush(account);
    }

    @Transactional(rollbackFor = Exception.class)
    public void changePassword(long accountId, String password) {
        Account existAccount = accountDao.findById(accountId).orElseThrow(() -> new IllegalArgumentException("找不到对应的账户(" + accountId + ")"));
        String accountSalt = existAccount.getSalt();
        if (StringUtils.isEmpty(accountSalt)) {
            if (StringUtils.isNotEmpty(existAccount.getEmail())) {
                accountSalt = existAccount.getEmail();
            } else if (StringUtils.isNotEmpty(existAccount.getTelPhone())) {
                accountSalt = existAccount.getTelPhone();
            } else if (StringUtils.isNotEmpty(existAccount.getUsername())) {
                accountSalt = existAccount.getUsername();
            }
        }

        String rawPassword = this.desPassword(password);
        existAccount.setRawPassword(rawPassword);
        String encryptPassword = CryptUtils.encryptPassword(accountSalt, rawPassword, this.salt);
        //密码一样跳过
        if (encryptPassword.equals(existAccount.getPassword())) {
            throw new IllegalArgumentException("新密码不能和原密码一样");
        }
        existAccount.setSalt(accountSalt);
        existAccount.setPassword(encryptPassword);
        //修改密码后重新登录，不重置密码标志
        accountDao.saveAndFlush(existAccount);
    }

    public List<AccountUserDTO> accountUserList(long accountId) {
        return accountUserExtendDao.findUserByAccountId(accountId);
    }

    /**
     * async update account.
     *
     * @param account
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateLastLoginTime(Long accountId) {
        Optional<Account> optionalAccount = accountDao.findById(accountId);
        if (!optionalAccount.isPresent()) {
            logger.warn("更新登录时间失败，accountId:{}", accountId);
            return;
        }
        Account account = optionalAccount.get();
        account.setLastLoginTime(Calendar.getInstance().getTime());
        accountDao.saveAndFlush(account);
    }

    public Account saveAndFlush(Account account) {
        account = accountDao.saveAndFlush(account);
        accountPubSubService.publish(account);
        return account;
    }

    public String getEncryptedPwd(Long accountId) {
        Optional<Account> account = accountDao.findById(accountId);
        if (account.isPresent()) {
            return account.get().getPassword();
        } else {
            return null;
        }
    }

    public void updateDoubleAuth(long accountId,
                                 boolean status) {
        Optional<Account> accountOptional = accountDao.findById(accountId);
        if (accountOptional.isPresent()) {
            Account account = accountOptional.get();
            account.setDoubleAuthFlag(status);
            accountDao.saveAndFlush(account);
        } else {
            throw new IllegalArgumentException("账号不存在！");
        }
    }

    public void updateBindAuth(long accountId,
                               boolean status) {
        Optional<Account> accountOptional = accountDao.findById(accountId);
        if (accountOptional.isPresent()) {
            Account account = accountOptional.get();
            account.setBindAuthFlag(status);
            accountDao.saveAndFlush(account);
        } else {
            throw new IllegalArgumentException("账号不存在！");
        }
    }
}
