package com.xforceplus.business.app.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.app.AppApi;
import com.xforceplus.api.model.AppModel.Request.Create;
import com.xforceplus.api.model.AppModel.Request.Query;
import com.xforceplus.api.model.AppModel.Request.Save;
import com.xforceplus.business.app.service.AppService;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.App;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.context.request.WebRequest;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.xforceplus.api.common.Uri.PATH_CURRENT_PREFIX;

/**
 * @author geewit
 */
@SuppressWarnings("all")
@Api(value = "应用相关接口", description = "应用相关接口")
@Validated
@Controller
public class AppController implements AppApi {
    private final static Logger logger = LoggerFactory.getLogger(AppController.class);


    private final AppService appService;

    public AppController(AppService appService) {
        this.appService = appService;
    }

    /**
     * 获取应用分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:business:app:read"})
    @ApiIgnore
    @ApiOperation(value = "获取应用分页", notes = "通过分页参数，获取应用分页")
    @ResponseBody
    @RequestMapping(name = "应用分页列表查询", value = {"/api/global/apps/query"}, method = RequestMethod.GET)
    public Page<App> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<App> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<App> page = appService.page(specification, pageable);
        return page;
    }

    @AuthorizedDefinition(resources = {"xforce:business:app:read"})
    @ApiOperation(value = "获取应用分页", notes = "通过分页参数，获取应用分页")
    @Override
    public ResponseEntity<Page<App>> page(@ApiParam(value = "request") Query query,
                                          @ApiParam(value = "pageable") Pageable pageable) {
        Page<App> result = appService.page(query, pageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:app:read"})
    @ApiIgnore
    @ApiOperation(value = "获取应用列表", notes = "通过参数，获取应用列表")
    @ResponseBody
    @RequestMapping(name = "应用列表", value = Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<App>> list() {
        List<App> result = appService.list();
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:app:save"})
    @ApiOperation(value = "新增应用")
    @Override
    public ResponseEntity<App> create(@ApiParam(value = "app", required = true) Create model) {
        App result = appService.create(model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:app:save"})
    @ApiOperation(value = "修改应用")
    @Override
    public ResponseEntity<App> update(@ApiParam(value = "appId", required = true) long appId,
                                      @ApiParam(value = "app", required = true) Save model) {
        App result = appService.update(appId, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:app:save"})
    @ApiOperation(value = "修改应用状态")
    @Override
    public ResponseEntity<String> updateStatus(@ApiParam(value = "appId", required = true) long appId,
                                               @ApiParam(value = "status", required = true) int status) {
        appService.updateStatus(appId, status);
        return ResponseEntity.ok("修改成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:app:read"})
    @ApiOperation(value = "根据id获取应用信息")
    @Override
    public ResponseEntity<App> info(long appId) {
        App result = appService.findById(appId);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:app:save"})
    @ApiOperation(value = "根据id删除帐号")
    @Override
    public ResponseEntity<String> delete(@ApiParam(value = "appId", required = true) long appId) {
        appService.deleteById(appId);
        return ResponseEntity.ok("删除成功");
    }


    @ApiOperation(value = "获取当前用户可管理app集合")
    @ResponseBody
    @RequestMapping(name = "获取当前用户可管理app集合", value = PATH_CURRENT_PREFIX + "/apps", method = RequestMethod.GET)
    public ResponseEntity<Set<Long>> currentApps() {
        Set<Long> appIds = appService.currentAppIds();
        return ResponseEntity.ok(appIds);
    }
}
