package com.xforceplus.business.app.service;

import com.xforceplus.api.model.AppModel.Request.Create;
import com.xforceplus.api.model.AppModel.Request.Query;
import com.xforceplus.api.model.AppModel.Request.Save;
import com.xforceplus.dao.AppDao;
import com.xforceplus.dao.UserAppDao;
import com.xforceplus.entity.App;
import com.xforceplus.query.AppQueryHelper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.core.utils.reflection.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Set;


/**
 * 帐号Service
 *
 * @author geewit
 */
@Service
public class AppService {

    private final AppDao appDao;

    private final UserAppDao userAppDao;

    public AppService(AppDao appDao, UserAppDao userAppDao) {
        this.appDao = appDao;
        this.userAppDao = userAppDao;

    }

    public App findOne(Query query) {
        Specification<App> specification = AppQueryHelper.querySpecification(query);
        return appDao.findOne(specification).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
    }

    public Page<App> page(Query query, Pageable pageable) {
        Specification<App> specification = AppQueryHelper.querySpecification(query);
        return appDao.findAll(specification, pageable);
    }

    public Page<App> page(Specification<App> specification, Pageable pageable) {
        return appDao.findAll(specification, pageable);
    }

    public List<App> list() {
        Query query = new Query();
        Specification<App> specification = AppQueryHelper.querySpecification(query);
        return appDao.findAll(specification, Sort.by("appId"));
    }

    public List<App> list(Specification<App> specification, Sort sort) {
        return appDao.findAll(specification, sort);
    }

    @Transactional(rollbackFor = Exception.class)
    public App create(Create model) {
        if (appDao.existsById(model.getAppId())) {
            throw new IllegalArgumentException("已存在该id");
        }
        App entity = new App();
        BeanUtils.copyProperties(model, entity);
        entity = appDao.saveAndFlush(entity);
        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public App update(long appId, Save model) {
        App existEntity = this.findById(appId);
        BeanUtils.copyProperties(model, existEntity);
        existEntity = appDao.saveAndFlush(existEntity);
        return existEntity;
    }

    public App findById(long appId) {
        return appDao.findById(appId).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long appId) {
        App app = this.findById(appId);
        appDao.deleteById(app.getAppId());
    }

    @Transactional(rollbackFor = Exception.class)
    public App updateStatus(long appId, int status) {
        App existEntity = this.findById(appId);
        existEntity.setStatus(status);
        existEntity = appDao.saveAndFlush(existEntity);
        return existEntity;
    }

    public Set<Long> userApps(long userId) {
        Set<Long> appIds = userAppDao.findAppIdByUserId(userId);
        return appIds;
    }

    public Set<Long> currentAppIds() {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Set<Long> appIds = userAppDao.findAppIdByUserId(authorizedUser.getId());
        return appIds;
    }
}
