package com.xforceplus.business.client.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.client.ClientApi;
import com.xforceplus.api.model.ClientModel;
import com.xforceplus.business.client.service.ClientSecretService;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.domain.client.ClientDto;
import com.xforceplus.entity.Client;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import io.geewit.core.utils.reflection.BeanUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.Min;
import java.util.List;

/**
 * @author geewit
 */
@Api(value = "client相关接口", tags = {"client"})
@Controller
public class ClientController implements ClientApi {
    private final ClientSecretService clientService;
    public ClientController(ClientSecretService clientService) {
        this.clientService = clientService;
    }

    /**
     * @param pageable
     * @return
     */
    @ResponseBody
    @GetMapping(name = "client分页列表", value = Path.PAGE)
    public ResponseEntity<Page<Client>> page(@SpringQueryMap ClientModel.Request.Query query, Pageable pageable) {
        Page<Client> result = clientService.findByQuery(query, pageable);
        return ResponseEntity.ok(result);
    }

    /**
     * @deprecated use {@link #listV2()} } ()} instead.
     */
    @ResponseBody
    @GetMapping(name = "client列表", value = Path.LIST)
    @Deprecated
    public List<Client> list() {
        return clientService.findAll();
    }

    @ResponseBody
    @GetMapping(name = "client列表", value = Path.LIST_v2)
    public ResponseEntity<List<Client>> listV2() {
        return ResponseEntity.ok(clientService.findAll());
    }

    /**
     * @deprecated use {@link #getV2(String)} ()} instead.
     */
    @ResponseBody
    @GetMapping(name = "获取client", value = Path.INFO)
    @Deprecated
    public ClientDto get(@PathVariable("clientId") String clientId) {
        return clientService.findById(clientId);
    }

    @Override
    public ResponseEntity<Client> getV2(@PathVariable("clientId") String clientId) {
        return ResponseEntity.ok(clientService.findById(clientId));
    }

    @Override
    public ResponseEntity<Client> check(String clientId, String secret) {
        Client result = clientService.findByClientIdAndSecret(clientId, secret);
        return ResponseEntity.ok(result);
    }

    /**
     * @deprecated use {@link #createV2(ClientDto)} ()} } ()} instead.
     */
    @ResponseBody
    @PostMapping(name = "新增client", value = Path.CREATE)
    @Deprecated
    public Client create(@RequestBody ClientDto clientForm) {
        Client client = new Client();
        BeanUtils.copyProperties(clientForm, client);
        client = clientService.create(client);
        return client;
    }
    @AuthorizedDefinition(resources = {"xforce:operation:client:save"})
    @ResponseBody
    @PostMapping(name = "新增client", value = Path.CREATE_V2)
    public ResponseEntity<Client> createV2(@RequestBody ClientDto clientForm) {
        Client client = new Client();
        BeanUtils.copyProperties(clientForm, client);
        client = clientService.create(client);
        return ResponseEntity.ok(client);
    }

    /**
     * @deprecated use {@link #updateV2(String, ClientDto)} ()} } ()} instead.
     */
    @ResponseBody
    @PutMapping(name = "更新client", value = Path.EDIT)
    @Deprecated
    public Client update(@PathVariable("clientId") String clientId, @RequestBody ClientDto clientForm) {
        Client client = new Client();
        BeanUtils.copyProperties(clientForm, client, new String[]{"clientId"});
        return clientService.update(clientId, client);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:client:save"})
    @ResponseBody
    @PutMapping(name = "更新client", value = Path.EDIT_V2)
    public ResponseEntity<Client> updateV2(@PathVariable("clientId") String clientId, @RequestBody ClientDto clientForm) {
        Client client = new Client();
        BeanUtils.copyProperties(clientForm, client, new String[]{"clientId"});
        return ResponseEntity.ok(clientService.update(clientId, client));
    }

    /**
     * 删除成功
     *
     * @param clientId ClientId
     * @return ResponseEntity<String>
     */
    @ResponseBody
    @DeleteMapping(name = "删除client", value = Path.DELETE)
    public ResponseEntity<String> delete(@PathVariable("clientId") String clientId) {
        clientService.delete(clientId);
        return ResponseEntity.ok("删除成功");
    }

    /**
     * clientId绑定用户ID
     *
     * @param clientId
     * @param userId
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:client:save"})
    @ApiOperation(value = "client绑定用户信息")
    @ResponseBody
    @PostMapping(name = "clientId绑定用户ID", value = Path.INFO + "/bind")
    public ResponseEntity<Client> bindUserId(@PathVariable(value = "clientId") String clientId,
                                              @RequestParam(value = "userId", required = true) @Min(value = 1L, message = "用户ID必须大于0") Long userId) {
        return ResponseEntity.ok(clientService.bindUserId(clientId, userId));
    }
    @AuthorizedDefinition(resources = {"xforce:operation:client:save"})
    @ApiOperation(value = "client解绑用户信息")
    @ResponseBody
    @PostMapping(name = "clientId解绑用户", value = Path.INFO + "/unbind")
    public ResponseEntity<Client> unbindUserId(@PathVariable(value = "clientId") String clientId) {
        return ResponseEntity.ok(clientService.unbindUserId(clientId));
    }

    @AuthorizedDefinition(resources = {"xforce:operation:client:secret"})
    @ApiOperation(value = "查询client密钥")
    @Override
    public ResponseEntity<String> getSecret(String clientId) {
        Client client = clientService.findById(clientId);
        return ResponseEntity.ok(client.getSecret());
    }
}
